import * as React from "react";

import Countdown from "react-countdown-now";
import { Delta } from "react-countdown-now";

import { AutoResolve as AutoResolveModel } from "aegis/models";
import { DisableAutoResolveButton } from "./disable-auto-resolve-button/container";
import { AutoResolveUser, UserType } from "./user";

import {
  AlignItems,
  Background,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  JustifyContent,
  Layout,
  Overflow,
  Position,
  StyledLayout,
  TextAlign,
  TextType,
  Tooltip,
  TooltipDirection,
  VerticalAlign
} from "twitch-core-ui";

import "./styles.scss";

export interface Props {
  autoResolve: AutoResolveModel;
  showCreatedBy?: boolean;
  borderBottom?: boolean;
}

export enum TestSelectors {
  Countdown = "countdown",
  Reporter = "reporter",
  ResolveCount = "resolve-count",
  Target = "target",
  Content = "content",
  Reason = "reason"
}

export const AutoResolve: React.SFC<Props> = ({ autoResolve, showCreatedBy, borderBottom }) => {
  const users: JSX.Element[] = [];
  if (autoResolve.targetUser) {
    users.push(
      <AutoResolveUser
        data-test-selector={TestSelectors.Target}
        key="target"
        user={autoResolve.targetUser}
        type={UserType.TARGET}
      />
    );
  }
  if (autoResolve.fromUser) {
    users.push(
      <AutoResolveUser
        data-test-selector={TestSelectors.Reporter}
        key="from"
        user={autoResolve.fromUser}
        type={UserType.REPORTER}
      />
    );
  }

  const addedBy = showCreatedBy ? (
    <Layout>
      <CoreText color={Color.Alt2} fontSize={FontSize.Size8} type={TextType.Span}>
        Added by{" "}
      </CoreText>
      <CoreText type={TextType.Strong} color={Color.Alt2} fontSize={FontSize.Size8}>
        {autoResolve.createdBy ? autoResolve.createdBy.ldap : "unknown"}
      </CoreText>
    </Layout>
  ) : null;

  return (
    <StyledLayout overflow={Overflow.Visible} className={"auto-resolve-rule"}>
      <Layout
        display={Display.Flex}
        alignItems={AlignItems.Center}
        justifyContent={JustifyContent.Around}
        overflow={Overflow.Visible}
        textAlign={TextAlign.Center}
        padding={{ x: 1 }}
        position={Position.Relative}
        fullWidth
      >
        <Layout flexShrink={0} flexGrow={0} textAlign={TextAlign.Center}>
          <Layout fullWidth padding={{ right: 1 }}>
            <Tooltip label="Total autoresolved reports" direction={TooltipDirection.Left}>
              <CoreText
                color={Color.Alt2}
                align={VerticalAlign.Middle}
                fontSize={FontSize.Size4}
                type={TextType.Strong}
                data-test-selector={TestSelectors.ResolveCount}
              >
                {autoResolve.count}
              </CoreText>
            </Tooltip>
          </Layout>
        </Layout>
        <Layout
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
          justifyContent={JustifyContent.Start}
          textAlign={TextAlign.Left}
          flexShrink={1}
          flexGrow={1}
          padding={{ top: 0.5 }}
        >
          {users}
          {addedBy}
          <Layout display={Display.Flex} flexDirection={FlexDirection.Column} margin={{ y: 0.5 }}>
            {autoResolve.content && (
              <div
                className="aegis-pill__content auto-resolve-rule__content_reason_pills"
                data-test-selector={TestSelectors.Content}
              >
                {autoResolve.content.displayName}
              </div>
            )}
            {autoResolve.reason && autoResolve.content && <Layout padding={{ top: 0.5 }} />}
            {autoResolve.reason ? (
              <div
                className="aegis-pill__reason auto-resolve-rule__content_reason_pills"
                data-test-selector={TestSelectors.Reason}
              >
                {autoResolve.reason.displayName}
              </div>
            ) : null}
          </Layout>
        </Layout>
        <Layout
          flexShrink={0}
          flexGrow={0}
          padding={{ left: 1 }}
          display={Display.Flex}
          justifyContent={JustifyContent.Center}
          data-test-selector={TestSelectors.Countdown}
        >
          <Countdown date={new Date(autoResolve.resolveUtil!)} daysInHours renderer={CountdownRenderer} />
        </Layout>

        <StyledLayout
          background={Background.Alt}
          display={Display.Flex}
          justifyContent={JustifyContent.Center}
          alignItems={AlignItems.Center}
          position={Position.Absolute}
          attachRight
          overflow={Overflow.Hidden}
          fullHeight
          className={"auto-resolve-rule-delete-button"}
        >
          <DisableAutoResolveButton id={autoResolve.id} />
        </StyledLayout>
      </Layout>
      <StyledLayout margin={{ x: 1 }} borderBottom={borderBottom} />
    </StyledLayout>
  );
};

// Custom countdown renderer to ignore hours
const CountdownRenderer = (delta: Delta) => {
  return (
    <CoreText>
      {("0" + delta.minutes).slice(-2)}:{("0" + delta.seconds).slice(-2)}
    </CoreText>
  );
};
