import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import * as errorNotifications from "aegis/functionality/error-notifications";
import { UserRole } from "aegis/models";
import { MockUser } from "aegis/models/mocks";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { BanFormComponent, CreateSuspensionStatus, SpecialUserRoles, TestSelectors } from "./component";
import { SuspensionGuideContentsOptions } from "./contents";
import { SuspensionGuideDetailedReasonsOptions } from "./detailed-reason";
import { mockedBanFormQueries } from "./mocks";
import { SuspensionGuideReasonsOptions } from "./reasons";

jest.mock("aegis/functionality/error-notifications", () => ({
  errorNotification: jest.fn()
}));

Enzyme.configure({ adapter: new Adapter() });

const mockCreateSuspensionMutationSuccessWarningPayload = {
  data: {
    createSuspension: {
      enforcementType: "warning",
      __typename: "CreateSuspensionPayload"
    }
  }
};

describe("Ban Form", () => {
  beforeEach(() => {
    jest.clearAllMocks();
  });

  it("renders without crashing", () => {
    const wrapper = Enzyme.mount(wrapWithGQLProvider(<BanFormComponent />, mockedBanFormQueries).wrappedComponent);
    expect(wrapper.find(BanFormComponent)).toBeTruthy();
  });

  it("shows special user check for users with at least one special role", () => {
    SpecialUserRoles.forEach(role => {
      const user = MockUser();
      user.roles = [role];
      const wrapper = Enzyme.shallow(<BanFormComponent />);

      wrapper.find({ "data-test-selector": TestSelectors.UserInput }).simulate("change", user);
      expect(wrapper.find({ "data-test-selector": TestSelectors.SpecialUserCheck }).length).toEqual(1);
    });
  });

  it("hides special user check for users without any special role", () => {
    const user = MockUser();
    user.roles = [UserRole.Developer, UserRole.Affiliate];
    const wrapper = Enzyme.shallow(<BanFormComponent targetUserID="id" />);

    wrapper.find({ "data-test-selector": TestSelectors.UserInput }).simulate("change", user);

    expect(wrapper.find({ "data-test-selector": TestSelectors.SpecialUserCheck }).length).toEqual(0);
  });

  it("changes Enforce button if user is already suspended", () => {
    const suspendedUser = MockUser();
    const wrapper = Enzyme.shallow(<BanFormComponent targetUserID="id" />);

    wrapper.setState({ createStatus: CreateSuspensionStatus.Succeeded });
    wrapper.find({ "data-test-selector": TestSelectors.UserInput }).simulate("change", suspendedUser);
    expect(wrapper.find({ "data-test-selector": TestSelectors.SuspendButton }).html()).toContain(
      "Add Additional Enforcement"
    );
  });

  it("changes Enforce button if user input is changed", () => {
    const suspendedUser = MockUser();
    const newTargetUser = MockUser({ isBanned: false, id: "newID" });

    const wrapper = Enzyme.shallow(<BanFormComponent targetUserID="id" />);
    wrapper.setState({ shouldShowEscalatePartnerCheck: false });
    wrapper.find({ "data-test-selector": TestSelectors.UserInput }).simulate("change", suspendedUser);
    expect(wrapper.find({ "data-test-selector": TestSelectors.SuspendButton }).html()).toContain(
      "Add Additional Enforcement"
    );

    wrapper.find({ "data-test-selector": TestSelectors.UserInput }).simulate("change", newTargetUser);
    expect(wrapper.find({ "data-test-selector": TestSelectors.SuspendButton }).html()).toContain("Enforce");
    expect(wrapper.state("createStatus")).toEqual(undefined);
  });

  it("changes Escalate button if user is escalate partner", () => {
    const suspendedUser = MockUser();
    const newTargetUser = MockUser({ isBanned: false, id: "newID" });

    const wrapper = Enzyme.shallow(<BanFormComponent targetUserID="id" />);
    wrapper.setState({ shouldShowEscalatePartnerCheck: true });
    wrapper.find({ "data-test-selector": TestSelectors.UserInput }).simulate("change", suspendedUser);
    expect(wrapper.find({ "data-test-selector": TestSelectors.SuspendButton }).html()).toContain("Escalate");

    wrapper.find({ "data-test-selector": TestSelectors.UserInput }).simulate("change", newTargetUser);
    expect(wrapper.find({ "data-test-selector": TestSelectors.SuspendButton }).html()).toContain("Escalate");
    expect(wrapper.state("createStatus")).toEqual(undefined);
  });

  it("disables suspend button if status succeeded", () => {
    const wrapper = Enzyme.shallow(<BanFormComponent targetUserID="id" />);
    wrapper.setState({ createStatus: CreateSuspensionStatus.Succeeded });
    expect(wrapper.find({ "data-test-selector": TestSelectors.SuspendButton }).prop("disabled")).toBe(true);
  });

  describe("warning is issued as result of enforcement", () => {
    it("when Block IP is checked, it displays an error notification", async () => {
      // stubbing the mutate prop
      const fakeMutate = jest.fn();
      fakeMutate.mockReturnValue(mockCreateSuspensionMutationSuccessWarningPayload);

      const wrapper = Enzyme.shallow(<BanFormComponent targetUserID="id" mutate={fakeMutate} />);
      const targetUser = MockUser();

      // Setup Ban Form
      // tslint:disable-next-line: no-any
      const instance = wrapper.instance() as any;
      instance.store._ipBan = true;
      instance.store._content = {};
      instance.store._reason = {};
      instance.store._detailedReason = {};

      await wrapper
        .find({ "data-test-selector": TestSelectors.UserInput })
        .first()
        .simulate("change", targetUser);

      await wrapper.find({ "data-test-selector": TestSelectors.SuspendButton }).simulate("click");

      expect(errorNotifications.errorNotification).toBeCalledWith(
        "Enforcement successful. IP ban failed because the user received a warning."
      );

      expect(
        wrapper
          .find({ "data-test-selector": TestSelectors.BlockIPError })
          .children()
          .text()
      ).toBe("IP not blocked because a warning was issued.");
    });
  });

  describe("disabled Suspend Button re-enables after form change", () => {
    let wrapper: Enzyme.ShallowWrapper;
    beforeEach(() => {
      wrapper = Enzyme.shallow(<BanFormComponent targetUserID="id" placeholders={{}} />);
      const suspendedUser = MockUser({ id: "different-user", isBanned: false, roles: [] });
      wrapper.find({ "data-test-selector": TestSelectors.UserInput }).simulate("change", suspendedUser);

      wrapper.find(SuspensionGuideContentsOptions).simulate("change", { description: "1234", code: 1 });
      wrapper.find(SuspensionGuideReasonsOptions).simulate("change", { description: "1234", code: 1 });
      wrapper.find(SuspensionGuideDetailedReasonsOptions).simulate("change", {
        code: 1,
        contentCode: "1",
        reasonCode: "1",
        strikeDetails: "qwer",
        strikeDescription: "asdf"
      });

      wrapper.setState({ createStatus: CreateSuspensionStatus.Succeeded });
      expect(wrapper.find({ "data-test-selector": TestSelectors.SuspendButton }).prop("disabled")).toBe(true);
    });

    it("Suspend Button re-enabled after user change", () => {
      const suspendedUser = MockUser({ id: "different-user-2", isBanned: false, roles: [] });
      wrapper.find({ "data-test-selector": TestSelectors.UserInput }).simulate("change", suspendedUser);

      expect(wrapper.find({ "data-test-selector": TestSelectors.SuspendButton }).prop("disabled")).toBe(false);
    });

    it("Suspend Button re-enabled after reason change", () => {
      const newDetailedReason = { code: 2, strikeDetails: "qwerasdf" };
      wrapper.find(SuspensionGuideDetailedReasonsOptions).simulate("change", newDetailedReason);

      expect(wrapper.find({ "data-test-selector": TestSelectors.SuspendButton }).prop("disabled")).toBe(false);
    });

    it("Suspend Button re-enabled after description change", () => {
      wrapper
        .find({ "data-test-selector": TestSelectors.Description })
        .simulate("change", { currentTarget: { value: "herp derping" } });

      expect(wrapper.find({ "data-test-selector": TestSelectors.SuspendButton }).prop("disabled")).toBe(false);
    });
  });
});
