import * as React from "react";

import { ApolloClient, ApolloQueryResult } from "apollo-client";
import gql from "graphql-tag";
import { observer } from "mobx-react";
import { ChildProps, compose, graphql, withApollo } from "react-apollo";

import { UserInput } from "aegis/functionality/components/user-input";
import { User, UserRole } from "aegis/models";

import { CheckBoxFixer } from "aegis/functionality/components/checkbox-fixer";
import { errorNotification } from "aegis/functionality/error-notifications";
import { FetchResult } from "apollo-link";
import {
  AlignContent,
  AlignItems,
  Button,
  CheckBox,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FlexWrap,
  FormGroup,
  FormGroupOrientation,
  JustifyContent,
  Layout,
  LoadingSpinner,
  Select,
  StyledLayout,
  SVG,
  SVGAsset,
  SVGType,
  TextArea,
  Tooltip,
  TooltipDirection
} from "twitch-core-ui";
import { EscalateStatus } from "../user/components/on-escalate-list";
import { ContentFields, DetailedReasonFields, FindEntry, ReasonFields } from "./common";
import { SuspensionGuideContent, SuspensionGuideContentsOptions, SuspensionGuideContentsResponse } from "./contents";
import {
  SuspensionGuideDetailedReason,
  SuspensionGuideDetailedReasonsOptions,
  SuspensionGuideDetailedReasonsResponse
} from "./detailed-reason";
import { BanformImageUploader } from "./image-uploader";
import { SuspensionGuideQuickSelect, SuspensionGuideQuickSelectsOptions } from "./quick-select";
import { SuspensionGuideReason, SuspensionGuideReasonsOptions, SuspensionGuideReasonsResponse } from "./reasons";
import { BanFormStore } from "./store";

export interface PublicProps {
  description?: string;
  targetUser?: string;
  targetUserID?: string;
  locationOfContent?: string;
  defaultContent?: string;
  defaultReason?: string;
  defaultDetailedReason?: string;
  placeholders?: Placeholders;
}

export interface Placeholders {
  reportID?: string;
  chatlog?: string;
  whisper?: string;
  locationOfContent?: string;
  screenshotLinks?: string;
  content?: string;
}

export enum CreateSuspensionStatus {
  NotStarted,
  Loading,
  Errored,
  Succeeded
}

export enum EnforcementType {
  Warning = "warning",
  Suspension = "suspension"
}

export enum TestSelectors {
  SpecialUserCheck = "special-user-check",
  UserInput = "user-input",
  SuspendButton = "suspend-button",
  Description = "description-textarea",
  BlockIPError = "block-ip-error"
}

export const SpecialUserRoles: UserRole[] = [
  UserRole.Partner,
  UserRole.Admin,
  UserRole.CEO,
  UserRole.EXEC,
  UserRole.Staff
];

interface ApolloProp {
  client?: ApolloClient<{}>;
}

type Props = PublicProps & ApolloProp;

class CreateSuspensionMutationData {
  targetUser: User;
  enforcementType: string;
}

class State {
  shouldShowReasonDescription?: boolean;
  shouldShowBanSpecialUserCheck?: boolean;
  shouldShowEscalatePartnerCheck: boolean;
  isBanSpecialUserChecked?: boolean;
  createStatus?: CreateSuspensionStatus;
  createSuspensionMessage?: React.ReactNode;
  blockIPError?: boolean;
}

type EntryQueryResponse = SuspensionGuideContentsResponse &
  SuspensionGuideReasonsResponse &
  SuspensionGuideDetailedReasonsResponse;

const defaultOptionsQuery = gql`
  query suspensionGuide($contentCode: String!, $reasonCode: String!) {
    suspensionGuideContents {
      ...ContentFields
    }
    suspensionGuideReasons(contentCode: $contentCode) {
      ...ReasonFields
    }
    suspensionGuideEntries(contentCode: $contentCode, reasonCode: $reasonCode) {
      ...DetailedReasonFields
    }
  }

  ${ContentFields}
  ${ReasonFields}
  ${DetailedReasonFields}
`;

export const createSuspension = gql`
  mutation createSuspension($input: CreateSuspensionInput!) {
    createSuspension(input: $input) {
      targetUser {
        id
        isBanned
      }
      enforcementType
    }
  }
`;

@observer
export class BanFormComponent extends React.Component<ChildProps<Props, CreateSuspensionMutationData>, State> {
  private store: BanFormStore;

  constructor(props: ChildProps<Props, CreateSuspensionMutationData>, context: {}) {
    super(props, context);

    // Store stores the suspension form data
    this.store = new BanFormStore();
    this.store.updatePlaceholders(props.placeholders!);

    // State maintains UI states
    this.state = {
      createStatus: CreateSuspensionStatus.NotStarted,
      shouldShowEscalatePartnerCheck: false
    };
  }

  // Preload defaults passed in
  public componentDidMount() {
    this.store.updateDefaultDescription(this.props.description!);
    this.store.updateDefaultLocationOfContent(this.props.locationOfContent!);

    if (this.props.defaultContent || this.props.defaultReason || this.props.defaultDetailedReason) {
      this.props
        .client!.query({
          query: defaultOptionsQuery,
          variables: {
            contentCode: this.props.defaultContent,
            reasonCode: this.props.defaultReason
          }
        })
        .then(
          (res: ApolloQueryResult<EntryQueryResponse>) => {
            // Field set order is important to not override each other
            this.updateSuspensionGuideContent(FindEntry(this.props.defaultContent!, res.data.suspensionGuideContents!));
            this.updateSuspensionGuideReason(FindEntry(this.props.defaultReason!, res.data.suspensionGuideReasons!));
            this.updateSuspensionGuideDetailedReason(
              FindEntry(this.props.defaultDetailedReason!, res.data.suspensionGuideEntries!)
            );
          },
          (res: ApolloQueryResult<{}>) => {
            console.error("Failed to fetch default options", res);
          }
        );
    }
  }

  updateDescription: React.FormEventHandler<HTMLTextAreaElement> = e => {
    this.store.updateDescription(e.currentTarget.value);
    this.setState({ createStatus: undefined });
  };

  updateTosBan: React.FormEventHandler<HTMLInputElement> = e => {
    this.store.updateTOSBan(e.currentTarget.checked);
  };

  updateIPBan: React.FormEventHandler<HTMLInputElement> = e => {
    this.store.updateIPBan(e.currentTarget.checked);
  };

  updateClearImages: React.FormEventHandler<HTMLInputElement> = e => {
    this.store.updateClearImages(e.currentTarget.checked);
  };

  updateSuspendAuthyMatch: React.FormEventHandler<HTMLInputElement> = e => {
    this.store.updateSuspendAuthyMatch(e.currentTarget.checked);
  };

  updateSuspendEscalatePartner: React.FormEventHandler<HTMLInputElement> = e => {
    this.store.updateSuspendEscalatePartner(e.currentTarget.checked);
  };

  updateContentLocationPlaceholder: React.FormEventHandler<HTMLSelectElement> = e => {
    const value = (e.target as HTMLSelectElement).value;
    this.store.updateDescriptionPlaceholder("ᐸLOCATION_OF_CONTENTᐳ", value);
    this.setState({ createStatus: undefined });
  };

  updateContentTypePlaceholder: React.FormEventHandler<HTMLSelectElement> = e => {
    const value = (e.target as HTMLSelectElement).value;
    this.store.updateDescriptionPlaceholder("ᐸTYPE_OF_CONTENTᐳ", value + " ");
    this.setState({ createStatus: undefined });
  };

  showSuspensionReason: React.MouseEventHandler<HTMLElement> = () => {
    this.setState({
      shouldShowReasonDescription: true
    });
  };

  updateIsBanSpecialUserChecked: React.FormEventHandler<HTMLInputElement> = e => {
    this.setState({
      isBanSpecialUserChecked: !!e.currentTarget.checked
    });
  };

  submitSuspension: React.MouseEventHandler<HTMLElement> = async () => {
    const { store, props } = this;

    this.setState({
      createStatus: CreateSuspensionStatus.Loading,
      createSuspensionMessage: null,
      blockIPError: false
    });

    try {
      const payload = await props.mutate!({
        variables: {
          input: {
            targetUserID: store.targetUser!.id,
            content: store.contentID,
            reason: store.reasonID,
            detailedReason: store.detailedReasonID,
            description: store.fullDescription,
            tosBan: store.tosBan,
            ipBan: store.ipBan,
            suspendAuthyMatch: store.suspendAuthyMatch,
            suspendEscalatedPartner: store.suspendEscalatePartner,
            clearImages: store.clearImages,
            points: store.detailedReason!.points
          }
        }
      });

      this.setState({
        createStatus: CreateSuspensionStatus.Succeeded,
        blockIPError: this.notifyOfEnforcement(payload)
      });
      this.store.updateTargetUser({ ...this.store.targetUser!, isBanned: true });
    } catch (error) {
      const errorMessages: React.ReactNode[] = [];
      const baseText =
        `Failed to suspend ${store.targetUser}. Please try again. ` +
        `If this problem persists, you can send this error message to Safety Oncall: `;

      errorMessages.push(<CoreText bold>{baseText}</CoreText>);

      if (error) {
        errorMessages.push(<CoreText color={Color.Error}>{error.toString()}</CoreText>);
      } else {
        errorMessages.push(<CoreText color={Color.Error}>Unknown Error occurred. Server returned null</CoreText>);
      }

      this.setState({
        createStatus: CreateSuspensionStatus.Errored,
        createSuspensionMessage: <Layout>{errorMessages}</Layout>
      });

      console.error("Failed to suspend user", error);
    } finally {
      // reset dangerous flags
      this.store.updateSuspendAuthyMatch(false);
      this.store.updateSuspendEscalatePartner(false);
    }
  };

  notifyOfEnforcement(payload: void | FetchResult): boolean {
    if (
      this.store.ipBan &&
      payload &&
      payload.data &&
      payload.data.createSuspension &&
      payload.data.createSuspension.enforcementType &&
      payload.data.createSuspension.enforcementType.toLowerCase() === EnforcementType.Warning
    ) {
      errorNotification("Enforcement successful. IP ban failed because the user received a warning.");
      return true;
    }
    return false;
  }

  render() {
    const { store, state } = this;

    let showStrikeReasonDescriptionButton: React.ReactNode = null;
    if (!state.shouldShowReasonDescription) {
      showStrikeReasonDescriptionButton = (
        <FormGroup label="" orientation={FormGroupOrientation.Horizontal}>
          <Tooltip
            label={
              !store.strikeDescription
                ? "This detailed reason does not have a description"
                : "Click to show detailed reason description"
            }
            direction={TooltipDirection.Right}
          >
            <Button
              data-track-click="ban-form-show-detailed-reason-description"
              onClick={this.showSuspensionReason}
              disabled={!store.strikeDescription}
            >
              Show detailed reason description
            </Button>
          </Tooltip>
        </FormGroup>
      );
    }

    let strikeReasonDescriptionComponent: React.ReactNode = null;
    if (state.shouldShowReasonDescription) {
      strikeReasonDescriptionComponent = (
        <FormGroup label="" orientation={FormGroupOrientation.Horizontal}>
          <TextArea value={store.strikeDescription} rows={8} />
        </FormGroup>
      );
    }

    let contentLocationComponent: React.ReactNode = null;
    if (store.description != null && store.description.indexOf("ᐸLOCATION_OF_CONTENTᐳ") !== -1) {
      contentLocationComponent = (
        <FormGroup
          label="Content Location"
          hint="select to replace ᐸLOCATION_OF_CONTENTᐳ"
          orientation={FormGroupOrientation.Horizontal}
        >
          <Select value={"ᐸLOCATION_OF_CONTENTᐳ"} onChange={this.updateContentLocationPlaceholder}>
            <option value="ᐸLOCATION_OF_CONTENTᐳ">Replaces ᐸLOCATION_OF_CONTENTᐳ</option>
            <option value="profile image">Profile Image</option>
            <option value="profile banner">Profile Banner</option>
            <option value="offline image">Offline Image</option>
          </Select>
        </FormGroup>
      );
    }

    let contentTypeComponent: React.ReactNode = null;
    if (store.description != null && store.description.indexOf("ᐸTYPE_OF_CONTENTᐳ") !== -1) {
      contentTypeComponent = (
        <FormGroup
          label="Content Type"
          hint="select to replace ᐸTYPE_OF_CONTENTᐳ"
          orientation={FormGroupOrientation.Horizontal}
        >
          <Select value={"ᐸTYPE_OF_CONTENTᐳ"} onChange={this.updateContentTypePlaceholder}>
            <option value="ᐸTYPE_OF_CONTENTᐳ">Replaces ᐸTYPE_OF_CONTENTᐳ</option>
            <option value="tv show">TV Show</option>
            <option value="movie">Movie</option>
            <option value="anime">Anime</option>
            <option value="live tv (soccer)">Soccer</option>
            <option value="sports">Sports</option>
          </Select>
        </FormGroup>
      );
    }

    let descriptionHint: React.ReactNode = null;
    if (store.detailedReason && !store.description) {
      descriptionHint = <CoreText color={Color.Error}> Description cannot be empty </CoreText>;
    }

    const additionalNotes: string[] = [];
    if (store.detailedReason) {
      if (store.detailedReason.strikeFormNote) {
        additionalNotes.push(store.detailedReason.strikeFormNote);
      }
      if (store.detailedReason.removeOffendingContent) {
        additionalNotes.push(
          "Make sure you have removed all content associated with this violation before you submit this form"
        );
      }
    }

    let additionalNotesList: React.ReactNode = null;
    if (additionalNotes.length !== 0) {
      additionalNotesList = (
        <Layout padding={{ bottom: 1 }}>
          <FormGroup label="Notes:">
            <CoreText color={Color.Error}>
              <ul>{additionalNotes.map((s, i) => <li key={i}>- {s}</li>)}</ul>
            </CoreText>
          </FormGroup>
        </Layout>
      );
    }

    return (
      <Layout margin={{ top: 2 }}>
        <StyledLayout border={true} margin={{ top: 2 }}>
          <BanformImageUploader
            onUploadComplete={this.store.addScreenshots}
            linksInDescription={this.store.linksInDescription}
            targetUserID={store.targetUser ? store.targetUser.id : undefined}
          />
        </StyledLayout>
        <StyledLayout border={true} margin={{ top: 2 }} padding={{ top: 2 }}>
          <StyledLayout padding={{ left: 1, right: 1 }}>
            <FormGroup label="Target User " orientation={FormGroupOrientation.Horizontal}>
              <UserInput
                name="ban-target-user"
                data-test-selector={TestSelectors.UserInput}
                defaultLogin={this.props.targetUser}
                defaultID={this.props.targetUserID}
                onChange={this.updateTargetUser}
              />
              {!!store.targetUser && (
                <EscalateStatus
                  userID={store.targetUser.id}
                  onCompleted={isEscalatePartner =>
                    this.setState({
                      shouldShowEscalatePartnerCheck: isEscalatePartner
                    })
                  }
                />
              )}
            </FormGroup>

            <Layout padding={{ top: 2, bottom: 2 }}>
              <Layout padding={{ bottom: 1 }} alignContent={AlignContent.Center} alignItems={AlignItems.Center}>
                <SuspensionGuideQuickSelectsOptions
                  onChange={this.updateSuspensionGuideQuickSelect}
                  value={store.quickSelectID}
                />
              </Layout>

              <SuspensionGuideContentsOptions
                onChange={this.updateSuspensionGuideContent}
                contentCode={store.contentID}
              />

              <SuspensionGuideReasonsOptions
                onChange={this.updateSuspensionGuideReason}
                contentCode={store.contentID}
                reasonCode={store.reasonID}
              />

              <SuspensionGuideDetailedReasonsOptions
                onChange={this.updateSuspensionGuideDetailedReason}
                contentCode={store.contentID}
                reasonCode={store.reasonID}
                detailedReasonCode={store.detailedReasonID}
              />

              {showStrikeReasonDescriptionButton}

              {strikeReasonDescriptionComponent}
            </Layout>

            <FormGroup label="Description" orientation={FormGroupOrientation.Horizontal}>
              <TextArea rows={6} value={store.descriptionPrefix} disabled={true} />
              <TextArea
                rows={10}
                value={store.description}
                onChange={this.updateDescription}
                data-test-selector={TestSelectors.Description}
              />

              {descriptionHint}

              {contentLocationComponent}

              {contentTypeComponent}
            </FormGroup>

            <Layout padding={{ top: 1 }}>
              <FormGroup label={"Actions"} orientation={FormGroupOrientation.Horizontal}>
                <Layout display={Display.Block}>
                  <Layout padding={{ bottom: 1 }}>
                    <Tooltip label="Use of this form currently requires a TOS Ban" direction={TooltipDirection.Right}>
                      <Layout padding={{ right: 0.5 }}>
                        <CheckBox
                          label="TOS Ban"
                          checked={!!store.tosBan}
                          onChange={this.updateTosBan}
                          disabled={true}
                        />
                      </Layout>
                    </Tooltip>
                  </Layout>

                  <Layout margin={{ bottom: 1 }}>
                    <CheckBoxFixer>
                      <CheckBox label="Block IP" checked={!!store.ipBan} onChange={this.updateIPBan} />
                      {this.state.blockIPError &&
                        this.state.createStatus === CreateSuspensionStatus.Succeeded &&
                        Boolean(store.ipBan) && (
                          <CoreText data-test-selector={TestSelectors.BlockIPError} color={Color.Error}>
                            {"IP not blocked because a warning was issued."}
                          </CoreText>
                        )}
                    </CheckBoxFixer>
                  </Layout>
                  <Layout margin={{ bottom: 1 }}>
                    <CheckBoxFixer>
                      <CheckBox label="Clear Images" checked={!!store.clearImages} onChange={this.updateClearImages} />
                    </CheckBoxFixer>
                  </Layout>

                  {this.state.shouldShowEscalatePartnerCheck && (
                    <Layout margin={{ bottom: 1 }}>
                      <CheckBox
                        label="Enforce the escalated partner"
                        checked={!!store.suspendEscalatePartner}
                        onChange={this.updateSuspendEscalatePartner}
                      />
                    </Layout>
                  )}

                  <Layout margin={{ bottom: 1 }}>
                    <CheckBoxFixer>
                      <CheckBox
                        label="Suspend Authy ID Matches"
                        checked={!!store.suspendAuthyMatch}
                        onChange={this.updateSuspendAuthyMatch}
                      />
                    </CheckBoxFixer>
                  </Layout>

                  {this.banSpecialUserCheckComponent()}
                </Layout>
              </FormGroup>
            </Layout>

            {additionalNotesList}

            <FormGroup label="" orientation={FormGroupOrientation.Horizontal}>
              <Layout
                display={Display.Flex}
                padding={{ top: 1, bottom: 1 }}
                flexDirection={FlexDirection.Row}
                flexWrap={FlexWrap.NoWrap}
                alignItems={AlignItems.Center}
                alignContent={AlignContent.Center}
                justifyContent={JustifyContent.Center}
                fullHeight
              >
                <Layout flexGrow={0} flexShrink={0} margin={{ right: 3 }}>
                  {this.renderEnforceUserButton()}
                </Layout>
                <Layout flexGrow={0} flexShrink={0} margin={{ right: 3 }}>
                  {this.renderSuspensionStatus()}
                </Layout>
                <Layout flexGrow={1}>{this.renderSuspensionMessage()}</Layout>
              </Layout>
            </FormGroup>
          </StyledLayout>
        </StyledLayout>
      </Layout>
    );
  }

  private updateTargetUser = (user?: User) => {
    const { store } = this;
    if (this.store.targetUser && user && this.store.targetUser.id === user.id) {
      // Skip update if user is the same
      return;
    }
    store.updateTargetUser(user);
    store.updateSuspendEscalatePartner(false);
    this.updateShowBanSpecialUserCheck(user);
  };

  private updateSuspensionGuideContent = (detail: SuspensionGuideContent | undefined) => {
    this.store.updateContent(detail ? detail : null);
    this.resetShowReasonDescription();
  };

  private updateSuspensionGuideReason = (detail: SuspensionGuideReason | undefined) => {
    this.store.updateReason(detail ? detail : null);
    this.resetShowReasonDescription();
  };

  private updateSuspensionGuideDetailedReason = (detail: SuspensionGuideDetailedReason | undefined) => {
    if (detail && this.store.detailedReasonID !== detail.code) {
      this.store.updateDetailedReason(detail);
      this.resetShowReasonDescription();
    }
  };

  private updateSuspensionGuideQuickSelect = (qs: SuspensionGuideQuickSelect) => {
    this.store.updateQuickSelect(qs);
    this.resetShowReasonDescription();
  };

  private updateShowBanSpecialUserCheck = (u?: User) => {
    if (
      u &&
      u.roles &&
      u.roles.some(role => {
        return SpecialUserRoles.includes(role);
      })
    ) {
      this.setState({
        shouldShowBanSpecialUserCheck: true
      });
    } else {
      this.setState({
        shouldShowBanSpecialUserCheck: false
      });
    }

    this.setState({
      isBanSpecialUserChecked: false,
      createStatus: undefined
    });
  };

  private resetShowReasonDescription = () => {
    this.setState({
      shouldShowReasonDescription: false,
      createStatus: undefined
    });
  };

  private banSpecialUserCheckComponent = (): React.ReactNode => {
    if (!this.state.shouldShowBanSpecialUserCheck) {
      return null;
    }
    return (
      <Layout margin={{ bottom: 1 }}>
        <CheckBox
          data-test-selector={TestSelectors.SpecialUserCheck}
          label="This user is a SPECIAL USER (staff/etc)! Check here to confirm that you wish to take action."
          checked={!!this.state.isBanSpecialUserChecked}
          onChange={this.updateIsBanSpecialUserChecked}
        />
      </Layout>
    );
  };

  private renderEnforceUserButton = (): React.ReactNode => {
    const { store, state } = this;

    const missingInput =
      !store.targetUser ||
      !store.detailedReason ||
      !store.description ||
      (state.shouldShowBanSpecialUserCheck && !state.isBanSpecialUserChecked);
    let buttonText: string;
    if (this.state.shouldShowEscalatePartnerCheck && !store.suspendEscalatePartner) {
      buttonText = "Escalate";
    } else if (
      (store.targetUser && store.targetUser.isBanned) ||
      state.createStatus === CreateSuspensionStatus.Succeeded
    ) {
      buttonText = "Add Additional Enforcement";
    } else {
      buttonText = "Enforce";
    }

    const disabled =
      missingInput || [CreateSuspensionStatus.Loading, CreateSuspensionStatus.Succeeded].includes(state.createStatus!);
    const buttonComponent = (
      <Button
        data-track-click="ban-form-submit-suspension"
        onClick={this.submitSuspension}
        disabled={disabled}
        data-test-selector={TestSelectors.SuspendButton}
      >
        {buttonText}
      </Button>
    );

    if (missingInput) {
      return (
        <Tooltip label="Please fill out required information" direction={TooltipDirection.Right}>
          {buttonComponent}
        </Tooltip>
      );
    }

    if (state.createStatus === CreateSuspensionStatus.Succeeded) {
      return (
        <Tooltip label="Edit user, reason, or description to unlock" direction={TooltipDirection.Top}>
          {buttonComponent}
        </Tooltip>
      );
    }

    return buttonComponent;
  };

  private renderSuspensionStatus = (): React.ReactNode => {
    const { state } = this;
    if (state.createStatus === CreateSuspensionStatus.Loading) {
      return <LoadingSpinner delay={0} />;
    }
    if (state.createStatus === CreateSuspensionStatus.Errored) {
      return <SVG asset={SVGAsset.Warning} type={SVGType.Alert} />;
    }
    if (state.createStatus === CreateSuspensionStatus.Succeeded) {
      return <SVG asset={SVGAsset.NotificationSuccess} type={SVGType.Success} />;
    }
    return null;
  };

  private renderSuspensionMessage = (): React.ReactNode => {
    return this.state.createSuspensionMessage;
  };
}

export const BanForm = compose(
  withApollo,
  graphql<CreateSuspensionMutationData, PublicProps>(createSuspension)
)(BanFormComponent);
