import * as React from "react";

import gql from "graphql-tag";
import { GraphqlQueryControls } from "react-apollo";
import { ChildProps, compose, graphql } from "react-apollo";

import { Color, CoreText, FormGroup, FormGroupOrientation, Select, TextType } from "twitch-core-ui";
import { ContentFields, FindEntry, SuspensionGuideEntry } from "../common";

export interface PublicProps {
  contentCode: string;
  onChange: (content: SuspensionGuideContent) => void;
}

export class SuspensionGuideContent extends SuspensionGuideEntry {
  description: string;
}

export interface SuspensionGuideContentsResponse {
  suspensionGuideContents?: SuspensionGuideContent[];
}

export const SuspensionGuideContentsQuery = gql`
  query suspensionGuideContents {
    suspensionGuideContents {
      ...ContentFields
    }
  }

  ${ContentFields}
`;

interface GraphQLProps {
  data: SuspensionGuideContentsResponse & GraphqlQueryControls;
}

type Props = PublicProps & GraphQLProps;

class State {}

export class SuspensionGuideContentComponent extends React.Component<
  ChildProps<Props, SuspensionGuideContentsResponse>,
  State
> {
  selectValue: React.FormEventHandler<HTMLSelectElement> = e => {
    const value = (e.target as HTMLSelectElement).value;
    this.notifyChange(value);
  };

  public render() {
    let contentOptions: React.ReactNode[];
    const { loading, suspensionGuideContents, error } = this.props.data;
    const { props } = this;

    if (error) {
      return (
        <CoreText type={TextType.Strong} color={Color.Error}>
          Failed to load suspension guide contents: {error.message}
        </CoreText>
      );
    }

    if (!suspensionGuideContents) {
      return null;
    }

    if (loading) {
      contentOptions = [
        <option key="Loading" value={""}>
          Loading...
        </option>
      ];
    } else {
      contentOptions = [
        <option key="Select" value={""}>
          Select one ...
        </option>
      ];

      contentOptions.push(
        ...suspensionGuideContents.map((c: SuspensionGuideContent) => (
          <option key={c.id} value={c.code}>
            {c.description}
          </option>
        ))
      );
    }

    return (
      <FormGroup label="Content" orientation={FormGroupOrientation.Horizontal}>
        <Select disabled={loading} onChange={this.selectValue} value={props.contentCode}>
          {contentOptions}
        </Select>
      </FormGroup>
    );
  }

  private notifyChange = (code: string) => {
    this.props.onChange(FindEntry(code, this.props.data.suspensionGuideContents!)!);
  };
}

export const SuspensionGuideContentsOptions: React.ComponentClass<PublicProps> = compose(
  graphql<SuspensionGuideContentsResponse, Props>(SuspensionGuideContentsQuery)
)(SuspensionGuideContentComponent);
