import * as React from "react";

import gql from "graphql-tag";
import { GraphqlQueryControls } from "react-apollo";
import { ChildProps, compose, graphql } from "react-apollo";

import { Color, CoreText, FormGroup, FormGroupOrientation, Select, TextType } from "twitch-core-ui";
import { DetailedReasonFields, FindEntry, SuspensionGuideEntry } from "../common";

export interface PublicProps {
  contentCode: string;
  reasonCode: string;
  detailedReasonCode: string;
  onChange: (reason: SuspensionGuideDetailedReason) => void;
}

export class SuspensionGuideDetailedReason extends SuspensionGuideEntry {
  contentCode: string;
  reasonCode: string;
  description: string;
  strikeDescription: string;
  strikeDetails: string;
  strikeFormNote: string;
  ipBan: boolean;
  permanent: boolean;
  clearImages: boolean;
  removeOffendingContent: boolean;
  points: number;
}

export interface SuspensionGuideDetailedReasonsResponse {
  suspensionGuideEntries?: SuspensionGuideDetailedReason[];
}

export const SuspensionGuideDetailedReasonsQuery = gql`
  query suspensionGuideEntries($contentCode: String!, $reasonCode: String!) {
    suspensionGuideEntries(contentCode: $contentCode, reasonCode: $reasonCode) {
      ...DetailedReasonFields
    }
  }

  ${DetailedReasonFields}
`;

interface GraphQLProps {
  data: SuspensionGuideDetailedReasonsResponse & GraphqlQueryControls;
}
export type Props = PublicProps & GraphQLProps;

export class SuspensionGuideDetailedReasonsComponent extends React.Component<
  ChildProps<Props, SuspensionGuideDetailedReasonsResponse>
> {
  selectValue: React.FormEventHandler<HTMLSelectElement> = e => {
    const value = (e.target as HTMLSelectElement).value;
    this.notifyChange(value);
  };

  public render() {
    let options: React.ReactNode[];
    const { loading, error } = this.props.data;
    const { suspensionGuideEntries } = this.props.data;
    const { props } = this;
    this.autoSelectSingleReason();

    if (error) {
      return (
        <CoreText type={TextType.Strong} color={Color.Error}>
          Failed to load suspension guide detailed reasons: {error.message}
        </CoreText>
      );
    }

    if (!suspensionGuideEntries) {
      return null;
    }

    if (loading) {
      options = [
        <option key="Loading" value={""}>
          Loading...
        </option>
      ];
    } else {
      options = [
        <option key="Select" value={""}>
          Select one ...
        </option>
      ];

      options.push(
        ...suspensionGuideEntries.map((e: SuspensionGuideDetailedReason) => (
          <option key={e.code} value={e.code}>
            ({e.points}) {e.description}
          </option>
        ))
      );
    }

    return (
      <FormGroup label="Detailed Reason" orientation={FormGroupOrientation.Horizontal}>
        <Select
          disabled={loading || suspensionGuideEntries.length === 0}
          onChange={this.selectValue}
          value={props.detailedReasonCode}
        >
          {options}
        </Select>
      </FormGroup>
    );
  }

  private notifyChange = (code: string) => {
    this.props.onChange(FindEntry(code, this.props.data.suspensionGuideEntries!)!);
  };

  private autoSelectSingleReason = () => {
    if (this.props.data && !this.props.data.loading) {
      if (this.props.data.suspensionGuideEntries && this.props.data.suspensionGuideEntries.length === 1) {
        this.props.onChange(this.props.data.suspensionGuideEntries[0]);
      }
    }
  };
}

export const SuspensionGuideDetailedReasonsOptions: React.ComponentClass<PublicProps> = compose(
  graphql<{ contentCode: string; reasonCode: string }, SuspensionGuideDetailedReasonsResponse, Props>(
    SuspensionGuideDetailedReasonsQuery
  )
)(SuspensionGuideDetailedReasonsComponent);
