import * as React from "react";

import gql from "graphql-tag";
import { GraphqlQueryControls } from "react-apollo";
import { ChildProps, compose, graphql } from "react-apollo";

import { Color, CoreText, FormGroup, FormGroupOrientation, Select, TextType } from "twitch-core-ui";
import { ContentFields, DetailedReasonFields, ReasonFields } from "../common";
import { SuspensionGuideContent } from "../contents";
import { SuspensionGuideDetailedReason } from "../detailed-reason";
import { SuspensionGuideReason } from "../reasons";

export interface PublicProps {
  value: string;
  onChange: (quickSelect: SuspensionGuideQuickSelect) => void;
}

interface GraphQLProps {
  data: Response & GraphqlQueryControls;
}

export type Props = PublicProps & GraphQLProps;

export interface SuspensionGuideQuickSelect {
  id: string;
  content: SuspensionGuideContent;
  reason: SuspensionGuideReason;
  detailedReason: SuspensionGuideDetailedReason;
}

class State {}

export interface Response {
  suspensionGuideQuickSelects?: SuspensionGuideQuickSelect[];
}

// Change detailed reason to fragment
export const SuspensionGuideQuickSelectsQuery = gql`
  query suspensionGuideQuickSelects {
    suspensionGuideQuickSelects {
      id
      detailedReason {
        ...DetailedReasonFields
      }
      reason {
        ...ReasonFields
      }
      content {
        ...ContentFields
      }
    }
  }

  ${ContentFields}
  ${ReasonFields}
  ${DetailedReasonFields}
`;

export class SuspensionGuideQuickSelectsComponent extends React.Component<ChildProps<Props, Response>, State> {
  selectValue: React.FormEventHandler<HTMLSelectElement> = e => {
    this.notifyChange((e.target as HTMLSelectElement).value);
  };

  public render() {
    let options: React.ReactNode[];
    const { loading, suspensionGuideQuickSelects, error } = this.props.data;

    if (error) {
      return (
        <CoreText type={TextType.Strong} color={Color.Error}>
          Failed to load suspension guide quick selects: {error.message}
        </CoreText>
      );
    }

    if (!suspensionGuideQuickSelects) {
      return null;
    }

    if (loading) {
      options = [
        <option key="Loading" value={""}>
          Loading...
        </option>
      ];
    } else {
      options = [
        <option key="Select" value={""}>
          Select one ...
        </option>
      ];

      options.push(
        ...suspensionGuideQuickSelects.map((q: SuspensionGuideQuickSelect) => (
          <option key={q.id} value={q.id}>
            {q.detailedReason.description}
          </option>
        ))
      );
    }

    return (
      <FormGroup label="Quick Select" orientation={FormGroupOrientation.Horizontal}>
        <Select
          disabled={loading || suspensionGuideQuickSelects.length === 0}
          value={this.props.value}
          onChange={this.selectValue}
        >
          {options}
        </Select>
      </FormGroup>
    );
  }

  private notifyChange = (id: string) => {
    if (this.props.data && this.props.data.suspensionGuideQuickSelects) {
      this.props.onChange(this.findEntry(id)!);
    }
  };

  private findEntry = (id: string) => {
    if (!id) {
      return null;
    }

    return this.props.data.suspensionGuideQuickSelects!.find((q: SuspensionGuideQuickSelect) => {
      return q.id === id;
    }, undefined);
  };
}

export const SuspensionGuideQuickSelectsOptions: React.ComponentClass<PublicProps> = compose(
  graphql<{}, Response, Props>(SuspensionGuideQuickSelectsQuery)
)(SuspensionGuideQuickSelectsComponent);
