import * as React from "react";

import gql from "graphql-tag";
import { GraphqlQueryControls } from "react-apollo";
import { ChildProps, compose, graphql } from "react-apollo";

import { Color, CoreText, FormGroup, FormGroupOrientation, Select, TextType } from "twitch-core-ui";
import { FindEntry, ReasonFields, SuspensionGuideEntry } from "../common";

export interface PublicProps {
  contentCode: string;
  reasonCode: string;
  onChange: (reason: SuspensionGuideReason) => void;
}

export class SuspensionGuideReason extends SuspensionGuideEntry {
  description: string;
}

export interface SuspensionGuideReasonsResponse {
  suspensionGuideReasons?: SuspensionGuideReason[];
}

export const SuspensionGuideReasonsQuery = gql`
  query suspensionGuideReasons($contentCode: String!) {
    suspensionGuideReasons(contentCode: $contentCode) {
      ...ReasonFields
    }
  }

  ${ReasonFields}
`;

interface GraphQLProps {
  data: SuspensionGuideReasonsResponse & GraphqlQueryControls;
}

type Props = PublicProps & GraphQLProps;

class State {}

export class SuspensionGuideReasonsComponent extends React.Component<
  ChildProps<Props, SuspensionGuideReasonsResponse>,
  State
> {
  selectValue: React.FormEventHandler<HTMLSelectElement> = e => {
    const value = (e.target as HTMLSelectElement).value;
    this.notifyChange(value);
  };

  public render() {
    let options: React.ReactNode[];
    const { loading, suspensionGuideReasons, error } = this.props.data;
    const { props } = this;

    if (error) {
      return (
        <CoreText type={TextType.Strong} color={Color.Error}>
          Failed to load suspension guide reasons: {error.message}
        </CoreText>
      );
    }

    if (!suspensionGuideReasons) {
      return null;
    }

    if (loading) {
      options = [
        <option key="Loading" value={""}>
          Loading...
        </option>
      ];
    } else {
      options = [
        <option key="Select" value={""}>
          Select one ...
        </option>
      ];

      options.push(
        ...suspensionGuideReasons.map((r: SuspensionGuideReason) => (
          <option key={r.id} value={r.code}>
            {r.description}
          </option>
        ))
      );
    }

    return (
      <FormGroup label="Reason" orientation={FormGroupOrientation.Horizontal}>
        <Select
          disabled={loading || suspensionGuideReasons.length === 0}
          onChange={this.selectValue}
          value={props.reasonCode}
        >
          {options}
        </Select>
      </FormGroup>
    );
  }

  private notifyChange = (code: string) => {
    this.props.onChange(FindEntry(code, this.props.data.suspensionGuideReasons!)!);
  };
}

export const SuspensionGuideReasonsOptions: React.ComponentClass<PublicProps> = compose(
  graphql<{ contentCode: string }, SuspensionGuideReasonsResponse, Props>(SuspensionGuideReasonsQuery)
)(SuspensionGuideReasonsComponent);
