import { action, computed, observable } from "mobx";

import { User } from "aegis/models";
import { Placeholders } from "./component";
import { SuspensionGuideContent } from "./contents";
import { SuspensionGuideDetailedReason } from "./detailed-reason";
import { SuspensionGuideQuickSelect } from "./quick-select";
import { SuspensionGuideReason } from "./reasons";

export class BanFormStore {
  @observable private _quickSelect: SuspensionGuideQuickSelect | null = null;
  @observable private _content: SuspensionGuideContent | null = null;
  @observable private _reason: SuspensionGuideReason | null = null;
  @observable private _detailedReason: SuspensionGuideDetailedReason | null = null;
  @observable private _targetUser: User | null = null;
  @observable private _description: string = "";
  @observable private _tosBan: boolean = true;
  @observable private _ipBan: boolean = false;
  @observable private _clearImages: boolean = false;
  @observable private _defaultDescription: string = "";
  @observable private _defaultLocationOfContent: string = "";
  @observable private _placeholders: Placeholders | null = null;
  @observable private _suspendAuthyMatch: boolean = false;
  @observable private _suspendEscalatePartner: boolean = false;

  // Base Properties
  @computed
  get quickSelect() {
    return this._quickSelect;
  }
  @computed
  get content() {
    return this._content;
  }
  @computed
  get reason() {
    return this._reason;
  }
  @computed
  get detailedReason() {
    return this._detailedReason;
  }
  @computed
  get targetUser() {
    return this._targetUser;
  }
  @computed
  get description() {
    return this._description;
  }
  @computed
  get tosBan() {
    return this._tosBan;
  }
  @computed
  get ipBan() {
    return this._ipBan;
  }
  @computed
  get clearImages() {
    return this._clearImages;
  }
  @computed
  get defaultDescription() {
    return this._defaultDescription;
  }
  @computed
  get defaultLocationOfContent() {
    return this._defaultLocationOfContent;
  }
  @computed
  get suspendAuthyMatch() {
    return this._suspendAuthyMatch;
  }
  @computed
  get suspendEscalatePartner() {
    return this._suspendEscalatePartner;
  }

  // Derived properties
  @computed
  get quickSelectID() {
    return this._quickSelect ? this._quickSelect.id : "";
  }
  @computed
  get contentID() {
    return this._content ? this._content.code : "";
  }
  @computed
  get reasonID() {
    return this._reason ? this._reason.code : "";
  }
  @computed
  get detailedReasonID() {
    return this._detailedReason ? this._detailedReason.code : "";
  }
  @computed
  get detailedReasonPoints() {
    return this._detailedReason ? this._detailedReason.points : 0;
  }
  @computed
  get descriptionPrefix() {
    return this.detailedReason == null
      ? "select a detailed reason..."
      : `Content: ${this.content!.code}\n` +
          `Reason: ${this.reason!.code}\n` +
          `IP Block: ${!!this.ipBan}\n` +
          "Suspension: " +
          (this.detailedReason.points >= 100 ? "indefinite" : `${this.detailedReason.points} Day(s)`) +
          "\n" +
          `Cleared Images: ${!!this.clearImages}\n` +
          "----------------------------------------";
  }
  @computed
  get fullDescription() {
    return this.descriptionPrefix + "\n\n" + this.description;
  }
  @computed
  get strikeDescription() {
    return this._detailedReason ? this._detailedReason.strikeDescription : "";
  }
  @computed
  get placeholders() {
    return this._placeholders;
  }
  @computed
  get linksInDescription() {
    let links;
    if (this._description) {
      links = this._description.match(/(https?:\/\/[^\s]+)/g);
    }
    return links ? links : [];
  }

  // Setters
  // UpdateQuickSelect updates quick select when user selects a new value
  @action
  updateQuickSelect = (quickSelect: SuspensionGuideQuickSelect): void => {
    if (!!quickSelect) {
      this._content = quickSelect.content;
      this._reason = quickSelect.reason;
      this.updateDetailedReason(quickSelect.detailedReason);
    }
    this._quickSelect = quickSelect;
  };
  // ClearQuickSelect should be used when content/reason/detailed reason changed
  @action
  clearQuickSelect = (): void => {
    this._quickSelect = null;
  };
  @action
  updateContent = (content: SuspensionGuideContent | null): void => {
    this._content = content;
    this.updateReason(null);
    this.updateDetailedReason(null);
    this.clearQuickSelect();
  };
  @action
  updateReason = (reason: SuspensionGuideReason | null): void => {
    this._reason = reason;
    this.updateDetailedReason(null);
    this.clearQuickSelect();
  };
  @action
  updateDetailedReason = (detailedReason: SuspensionGuideDetailedReason | null): void => {
    this._detailedReason = detailedReason;
    this.clearQuickSelect();

    if (detailedReason != null) {
      this.updateDescription(this.generateDescription(detailedReason.strikeDetails));
      this.updateDescription(this.replacePlaceholders()!);
      this.updateIPBan(detailedReason.ipBan);
      this.updateClearImages(detailedReason.clearImages);
    } else {
      this.updateDescription(this.defaultDescription);
    }
  };
  @action
  updateTargetUser = (targetUser?: User): void => {
    this._targetUser = targetUser ? targetUser : null;
  };
  @action
  updateDescription = (description: string): void => {
    this._description = description;
  };
  @action
  updateDescriptionPlaceholder = (placeholder: string, value: string): void => {
    this.updateDescription(this.description.replace(placeholder, value));
  };
  @action
  updateTOSBan = (tosBan: boolean): void => {
    this._tosBan = tosBan;
  };
  @action
  updateIPBan = (ipBan: boolean): void => {
    this._ipBan = ipBan;
  };
  @action
  updateClearImages = (clearImages: boolean): void => {
    this._clearImages = clearImages;
  };
  @action
  updateSuspendAuthyMatch = (suspendAuthyMatch: boolean): void => {
    this._suspendAuthyMatch = suspendAuthyMatch;
  };
  @action
  updateSuspendEscalatePartner = (suspendEscalatePartner: boolean): void => {
    this._suspendEscalatePartner = suspendEscalatePartner;
  };
  @action
  updateDefaultDescription = (defaultDescription: string): void => {
    this._defaultDescription = defaultDescription;
  };
  @action
  updateDefaultLocationOfContent = (defaultLocationOfContent: string): void => {
    this._defaultLocationOfContent = defaultLocationOfContent;
  };
  @action
  updatePlaceholders = (placeholders: Placeholders): void => {
    this._placeholders = placeholders;
  };

  @action
  addScreenshots = (screenshotLinks: string): void => {
    if (this._placeholders && this._placeholders.screenshotLinks) {
      if (!this._placeholders.screenshotLinks.includes(screenshotLinks)) {
        this._placeholders = {
          ...this._placeholders,
          screenshotLinks: this.placeholders!.screenshotLinks + "\n\n" + screenshotLinks
        };
      }
      // Prepend screenshots to front of description if there were already screenshots uploaded
      this.updateDescription(this.description + "\n\n" + screenshotLinks);
    } else {
      // Replace placeholders if screenshots weren't uploaded yet
      this._placeholders = { ...this._placeholders, screenshotLinks: screenshotLinks };
      this.updateDescription(this.replacePlaceholders()!);
    }
  };

  // Public functions that should be private but is public for test
  replacePlaceholders = () => {
    let description = this.description;
    const placeholders = this.placeholders;

    if (!placeholders) {
      return;
    }

    description = this.updatePlaceholder(description, "ᐸCHATLOG_COPY_PASTAᐳ", placeholders.chatlog);
    description = this.updatePlaceholder(description, "ᐸWHISPERLOG_COPY_PASTAᐳ", placeholders.whisper);
    description = this.updatePlaceholder(
      description,
      "ᐸORIG_REPORT_CASEIDᐳ",
      placeholders.reportID,
      "original case ID: "
    );
    description = this.updatePlaceholder(description, "ᐸLOCATION_OF_CONTENTᐳ", placeholders.locationOfContent);
    description = this.updatePlaceholder(description, "ᐸSCREENSHOT_LINKᐳ", placeholders.screenshotLinks);
    description = this.updatePlaceholder(description, "ᐸCONTENT_COPY_PASTAᐳ", placeholders.content);

    return description;
  };

  private updatePlaceholder = (description: string, placeholder: string, value?: string, defaultPrefix?: string) => {
    if (!value) {
      return description;
    }

    if (description.includes(placeholder)) {
      return description.replace(placeholder, value);
    }

    // Append the place holder value if it's not in the template
    return description + "\n\n" + (defaultPrefix ? defaultPrefix : "") + value;
  };

  // Private functions
  // generateDescription generates description based on tempalte and default description
  private generateDescription = (descriptionTemplate: string) => {
    let desc = descriptionTemplate + this.defaultDescription;

    // Check for uploaded images ᐸSCREENSHOT_LINKᐳ
    if (
      desc.match(/https:\/\/s3\.amazonaws\.com\/uploads\.hipchat.com\/51098\/[0-9]+\/.*\/.*\.[a-z]{3}/i) || // <-- Hipchat file upload
      desc.match(/https:\/\/slack\-files\.com\/files\-pub\/.*\/.*\.[a-z]{3}/i) || // <-- Slack file upload
      desc.match(
        /https:\/\/s3\-us\-west\-1\.amazonaws\.com\/ttv\-moderation\/screenshots\/[a-z0-9]{16}\/.*\-\-.*\.[a-z]{3,4}/i
      ) || // <-- Twitch S3 bucket 1
      desc.match(
        /https:\/\/leviathan\-prod\.s3\-us\-west\-2\.amazonaws\.com\/file\-uploads\/[a-z0-9]{16}\/.*--.*\.[a-z]{3,4}/i
      ) || // <-- Twitch S3 bucket 2
      desc.match(/https:\/\/leviathan\.internal\.twitch\.tv\/file\-uploads\/[a-z0-9]{16}\/.*--.*\.[a-z]{3,4}/i) // <-- Leviathan
    ) {
      const screenshotPlaceholder = "ᐸSCREENSHOT_LINKᐳ";
      desc = desc.replace(screenshotPlaceholder, "");
    }

    // Check for original report id ᐸORIG_REPORT_CASEIDᐳ
    if (desc.match(/Original Report ID: [0-9]+/)) {
      const originalID = this.defaultDescription.match(/Original Report ID: ([0-9]+)/);
      desc = desc.replace(/Original Report ID: [0-9]+[\r]?[\n]?/, "");
      desc = desc.replace("ᐸORIG_REPORT_CASEIDᐳ", originalID![1]);
    }

    // Check for location of content ᐸLOCATION_OF_CONTENTᐳ and <title / bio>
    if (this.defaultLocationOfContent) {
      desc = desc.replace("<title / bio>", this.defaultLocationOfContent);
      desc = desc.replace("ᐸLOCATION_OF_CONTENTᐳ", this.defaultLocationOfContent);
    }

    // Check for contents ᐸCONTENT_COPY_PASTAᐳ
    if (this.defaultDescription) {
      desc = desc.replace("ᐸCONTENT_COPY_PASTAᐳ", "");
    }

    // Check for chatlog ᐸCHATLOG_COPY_PASTAᐳ
    if (
      desc.match(/\b[0-9]{4}\-[01][0-9]\-[0123][0-9]\b\s\b[01][0-9]:[0-5][0-9]:[0-5][0-9]\b\s\b[AP]M\b/i) || // <-- Leviathan chat logs
      desc.match(/\(#[a-z0-9_]+\) [a-z0-9_]+: /i) // <-- George's live feed
    ) {
      desc = desc.replace("ᐸCHATLOG_COPY_PASTAᐳ", "");
    }

    // Check for whisper logs ᐸWHISPERLOG_COPY_PASTAᐳ
    if (
      desc.match(/\b[0-9]{4}\-[01][0-9]\-[0123][0-9] [012][0-9]:[0-5][0-9]:[0-5][0-9] [a-z]+\t[a-z0-9_]+\t/i) || // <-- Whisper logs
      desc.match(/[0-9]{4}\-[01][0-9]\-[0123][0-9]T[012][0-9]:[0-5][0-9]:[0-5][0-9](\.[0-9]{3})?Z/) // <-- New whisper logs format
    ) {
      desc = desc.replace("ᐸWHISPERLOG_COPY_PASTAᐳ", "");
    }

    return desc;
  };
}
