import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";
import React = require("react");
import { Balloon, Button, FormGroup, Interactable, Tag } from "twitch-core-ui";
import { BountyBoardSelectTagPresentation, PublicProps } from "./component";

Enzyme.configure({ adapter: new Adapter() });

function defaultPropsGenerator(overrides?: Partial<PublicProps>): PublicProps {
  return {
    label: "l1",
    buttonLabel: "l2",
    value: [],
    options: ["t1", "t2", "t3"],
    onChange: jest.fn(),
    ...overrides
  };
}

const setupShallow = (overrides?: Partial<PublicProps>) => {
  const props = defaultPropsGenerator();
  return Enzyme.shallow(<BountyBoardSelectTagPresentation {...props} {...overrides} />);
};

describe("BountyBoardSelectTag", () => {
  it("renders not tags when empty list", () => {
    const wrapper = setupShallow({ value: [] });
    expect(wrapper.find(Tag)).not.toBeNull();
  });

  it("renders tags when list has options", () => {
    const wrapper = setupShallow({
      value: ["t1", "t2"]
    });
    expect(wrapper.find(Tag)).toHaveLength(2);
  });

  it("removes tag when clicking on them", () => {
    const onChangeSpy = jest.fn();
    const wrapper = setupShallow({
      value: ["t1", "t2"],
      onChange: onChangeSpy
    });
    expect(wrapper.find(Tag)).toHaveLength(2);
    wrapper
      .find(Tag)
      .at(0)
      .prop("onClick")();
    expect(onChangeSpy).toHaveBeenCalledWith(["t2"]);
  });

  it("renders dropdown with options", () => {
    const wrapper = setupShallow();
    wrapper.setState({ open: true });
    wrapper.update();
    const balloon = wrapper.find(Balloon);
    expect(balloon.find(Interactable)).toHaveLength(3);
  });

  it("renders sets state.open to true when dropdown clicked", () => {
    const wrapper = setupShallow();
    expect(wrapper.state("open")).toBeFalsy();
    wrapper.find(Button).simulate("click");
    wrapper.update();
    expect(wrapper.state("open")).toBeTruthy();
  });

  it("renders no dropdown when all tags selected", () => {
    const wrapper = setupShallow({ value: ["t1", "t2", "t3"] });
    expect(wrapper.find(Button)).not.toBeNull();
  });

  it("sends change event after adding text", () => {
    const onChangeSpy = jest.fn();
    const wrapper = setupShallow({ onChange: onChangeSpy });
    wrapper.setState({ open: true });
    wrapper.update();
    wrapper
      .find(Balloon)
      .find(Interactable)
      .at(0)
      .simulate("click");
    expect(onChangeSpy).toHaveBeenCalledWith(["t1"]);
  });

  it("hides select and sets tags disabled if disabled", () => {
    const wrapper = setupShallow({
      value: ["t1"],
      disabled: true
    });
    expect(wrapper.find(Tag).prop("disabled")).toBe(true);
    expect(wrapper.find(Button)).not.toBeNull();
  });

  it("renders labels", () => {
    const wrapper = setupShallow();
    expect(
      wrapper
        .find(Button)
        .render()
        .text()
    ).toBe("l2");
    expect(wrapper.find(FormGroup).prop("label")).toBe("l1");
  });
});
