import { ClickOutDetector } from "aegis/functionality/components/click-out-detector";
import * as React from "react";
import {
  Balloon,
  Button,
  ButtonType,
  Display,
  FormGroup,
  Interactable,
  InteractableType,
  Layout,
  Position,
  Tag,
  TagAction
} from "twitch-core-ui";

export interface PublicProps {
  label: string;
  buttonLabel: string;
  value: string[];
  options: string[];
  disabled?: boolean;
  optionToLabel?: (option: string) => string;
  onChange: (options: string[]) => void;
}

interface State {
  open: boolean;
}

type Props = PublicProps;

export class BountyBoardSelectTagPresentation extends React.Component<Props, State> {
  public state: State = {
    open: false
  };

  public render() {
    const { disabled, buttonLabel, label, options, value } = this.props;
    const unselectedOptions = options.filter(option => !value.includes(option));
    const { open } = this.state;
    return (
      <FormGroup label={label}>
        <Layout margin={{ bottom: 0.5 }}>{value.map(this.renderOption)}</Layout>
        {unselectedOptions.length > 0 &&
          !disabled && (
            <>
              <Button
                data-track-click={`bounty-board-select-tag-open-${label}`}
                disabled={disabled}
                dropdown
                onClick={this.onClickAddButton}
                type={ButtonType.Hollow}
              >
                {buttonLabel}
              </Button>
              {open && this.renderFilterBalloon(unselectedOptions)}
            </>
          )}
      </FormGroup>
    );
  }

  private renderFilterBalloon = (unselectedOptions: string[]) => {
    const { optionToLabel } = this.props;
    const platformsEl = unselectedOptions.map(option => (
      <Interactable key={option} onClick={this.onClickOption(option)} type={InteractableType.Alpha}>
        <Layout padding={{ x: 1, y: 0.5 }}>{optionToLabel ? optionToLabel(option) : option}</Layout>
      </Interactable>
    ));
    return (
      <Layout position={Position.Relative}>
        <ClickOutDetector onClickOut={this.onClickDismissSelect}>
          <Balloon show noTail offsetY="0">
            <Layout>{platformsEl}</Layout>
          </Balloon>
        </ClickOutDetector>
      </Layout>
    );
  };

  private renderOption = (option: string) => {
    const { disabled, optionToLabel } = this.props;
    return (
      <Layout key={option} display={Display.InlineBlock} margin={{ right: 0.5, bottom: 0.5 }}>
        <Tag
          action={TagAction.Remove}
          blurAfterClick
          disabled={disabled}
          label={optionToLabel ? optionToLabel(option) : option}
          onClick={this.onClickRemoveOption(option)}
        />
      </Layout>
    );
  };

  private onClickAddButton = () => {
    this.setState({ open: true });
  };

  private onClickDismissSelect = (e: MouseEvent) => {
    e.stopImmediatePropagation();
    this.setState({ open: false });
  };

  private onClickOption = (option: string) => {
    return () => {
      const { value, options } = this.props;
      const newValue = value.slice(0);
      newValue.push(option);
      if (newValue.length === options.length) {
        this.setState({ open: false });
      }
      this.props.onChange(newValue);
    };
  };

  private onClickRemoveOption = (removeOption: string) => {
    return () => {
      const newValue = this.props.value.filter(option => option !== removeOption);
      this.props.onChange(newValue);
    };
  };
}

export const BountyBoardSelectTag = BountyBoardSelectTagPresentation as React.ComponentClass<PublicProps>;
