import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { ConfirmationModalConfirmSelector } from "aegis/functionality/components/confirmation-modal/component";
import { mockErrorResponse, mockSuccessResponse } from "aegis/tests/apollo";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { waitSomeMs } from "aegis/tests/utils/wait-some-ms";
import { Button, ButtonType } from "twitch-core-ui";
import { BountyModerationStatusOption } from "../../models";
import { Props, ReviewBountyButton, TestSelectors } from "./component";

import { ReviewBounty } from "aegis/features/bounties/components/bounties-table-row-button/component";

Enzyme.configure({ adapter: new Adapter() });

const mockReviewBountyMutationSuccess = mockSuccessResponse(
  ReviewBounty,
  {
    input: {
      bountyID: "test_bounty_id",
      moderationStatus: BountyModerationStatusOption.APPROVE
    }
  },
  {
    data: {
      reviewPendingBounty: {
        bounty: {
          id: "test_bounty_id",
          moderationStatus: "Pass",
          lastModeratedTime: new Date(),
          __typename: "PendingBounty"
        },
        __typename: "ReviewPendingBountyPayloadResolver"
      }
    }
  }
);

const mockReviewBountyMutationError = mockErrorResponse(ReviewBounty, {
  input: { bountyID: "test_bounty_id_fail", moderationStatus: BountyModerationStatusOption.APPROVE }
});

const mocks = [mockReviewBountyMutationSuccess, mockReviewBountyMutationError];

function defaultPropsGenerator(overrides?: Partial<Props>): Props {
  return {
    bountyID: "test_bounty_id",
    reason: null,
    moderationStatus: BountyModerationStatusOption.APPROVE,
    action: "Approve",
    disabled: false,
    type: ButtonType.Success,
    onButtonClick: jest.fn(),
    onError: jest.fn(),
    ...overrides
  };
}

function setupMount(overrides?: Partial<Props>) {
  const props = defaultPropsGenerator(overrides);
  return Enzyme.mount(wrapWithGQLProvider(<ReviewBountyButton {...props} {...overrides} />, mocks).wrappedComponent);
}

describe("ReviewBountyButton", () => {
  const modalRoot = document.createElement("div");
  modalRoot.setAttribute("id", "portal-modal-root");
  const body = document.querySelector("body");
  if (body) {
    body.appendChild(modalRoot);
  }

  const globalDiv = document.createElement("div");
  document.body.appendChild(globalDiv);

  it("disables button if given disabled prop", () => {
    const wrapper = setupMount({ disabled: true });
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(true);
  });

  it("button click with success response update button type to success", async () => {
    const onError = jest.fn();
    const onButtonClick = jest.fn();
    const wrapper = setupMount({ onButtonClick: onButtonClick, onError: onError });
    wrapper
      .find({ "data-test-selector": TestSelectors.Button })
      .find(Button)
      .simulate("click");

    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");
    await waitSomeMs();
    wrapper.update();

    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("type")
    ).toBe(ButtonType.Success);

    expect(onButtonClick).toHaveBeenCalled();
    expect(onError).not.toHaveBeenCalled();
  });

  it("calls onError prop if a mutation failed", async () => {
    const onError = jest.fn();
    const onButtonClick = jest.fn();
    const wrapper = setupMount({
      bountyID: "test_bounty_id_fail",
      onError,
      onButtonClick
    });

    wrapper
      .find({ "data-test-selector": TestSelectors.Button })
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");
    await waitSomeMs();
    wrapper.update();

    expect(onError).toHaveBeenCalled();
    expect(onButtonClick).toHaveBeenCalled();
  });
});
