import { WithConfirmationModal } from "aegis/functionality/components/confirmation-modal";
import * as React from "react";

import gql from "graphql-tag";
import { Mutation } from "react-apollo";

import {
  BountyFailReason,
  BountyModerationStatusOption,
  ReviewPendingBountyInput,
  ReviewPendingBountyResponse
} from "aegis/features/bounties/models";
import { Button, ButtonType } from "twitch-core-ui";

export enum TestSelectors {
  Button = "review-bounty-button"
}

export interface PublicProps {
  bountyID: string;
  reason: BountyFailReason | null;
  moderationStatus: BountyModerationStatusOption;
  onError: () => void;
  onButtonClick: () => void;
  action: string;
  disabled?: boolean;
  type: ButtonType;
}

export type Props = PublicProps;

export const ReviewBountyButton: React.FC<Props> = ({
  disabled,
  bountyID,
  action,
  moderationStatus,
  reason,
  type,
  onButtonClick,
  onError
}) => {
  return (
    <ReviewBountyMutation mutation={ReviewBounty}>
      {mutation => {
        const mutationVars: ReviewPendingBountyInput = {
          input: {
            bountyID,
            moderationStatus
          }
        };
        if (moderationStatus === BountyModerationStatusOption.REJECT && reason) {
          mutationVars.input.reason = reason;
        }
        const reviewBounty = () => {
          onButtonClick();
          mutation({ variables: mutationVars }).catch(() => {
            onError();
          });
        };
        return (
          <WithConfirmationModal
            title={`${action} Bounty`}
            body={`Are you sure to ${action} this bounty?`}
            confirmLabelOverride={action}
            onConfirm={reviewBounty}
          >
            {confirm => (
              <Button
                disabled={disabled}
                type={type}
                onClick={confirm}
                data-test-selector={TestSelectors.Button}
                data-track-click={`bounty-board-review-${action}`}
              >
                {action}
              </Button>
            )}
          </WithConfirmationModal>
        );
      }}
    </ReviewBountyMutation>
  );
};

export class ReviewBountyMutation extends Mutation<ReviewPendingBountyResponse, ReviewPendingBountyInput> {}

export const ReviewBounty = gql`
  mutation reviewBounty($input: ReviewPendingBountyInput!) {
    reviewPendingBounty(input: $input) {
      bounty {
        id
        moderationStatus
        lastModeratedTime
      }
    }
  }
`;
