import * as React from "react";
import * as ReactMarkdown from "react-markdown";

import { BountyCampaign, BountyCampaignType, BountyCampaignVideo } from "aegis/features/bounties/models";
import { ModalCloser } from "aegis/functionality/components/modal-closer";
import {
  AlignContent,
  Background,
  Button,
  Color,
  CoreLink,
  CoreText,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  StyledLayout,
  Table,
  TableBody,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
  TextAlign,
  TextType
} from "twitch-core-ui";

export enum TestSelectors {
  TitleColumn = "bounties-table-row-campaign-title-column",
  SponsorColumn = "bounties-table-row-campaign-sponsor-column",
  TypeColumn = "bounties-table-row-campaign-type-column",
  StreamLengthSecondsColumn = "bounties-table-row-campaign-stream-length-seconds",
  PlatformsColumn = "bounties-table-row-campaign-platforms-column",
  DetailsColumn = "bounties-table-row-campaign-details-column",
  GameColumn = "bounties-table-row-campaign-game-column",
  VideoColumn = "bounties-table-row-campaign-video-column",
  ErrorText = "ErrorText"
}

export interface PublicProps {
  campaign: BountyCampaign;
  onCancel: () => void;
}

export type Props = PublicProps;

export class BountyCampaignDetailsComponent extends React.Component<Props> {
  public render() {
    const { campaign, onCancel } = this.props;

    return (
      <StyledLayout background={Background.AccentAlt} padding={{ x: 0.5, y: 0.5 }} fullHeight fullWidth>
        <StyledLayout
          background={Background.Base}
          fullHeight={true}
          alignContent={AlignContent.Center}
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
        >
          <ModalCloser closeModal={onCancel} />
          <Layout padding={{ y: 1 }} textAlign={TextAlign.Center}>
            <CoreText type={TextType.H3}>Campaign Details</CoreText>
          </Layout>
          <StyledLayout padding={{ y: 2 }}>
            <Table>
              <TableHeader>
                <TableHeading label="Title" />
                <TableHeading label="Sponsor" />
                <TableHeading label="Type" />
                <TableHeading label="Required Length Seconds" />
                <TableHeading label="Platforms" />
                <TableHeading label="Details" />
                <TableHeading label="Game" />
                <TableHeading label="Video" />
              </TableHeader>
              <TableBody>
                <TableRow>
                  <TableCell data-test-selector={TestSelectors.TitleColumn}>{this.renderTitle()}</TableCell>
                  <TableCell data-test-selector={TestSelectors.SponsorColumn}>{campaign.sponsor}</TableCell>
                  <TableCell data-test-selector={TestSelectors.TypeColumn}>{campaign.type}</TableCell>
                  <TableCell data-test-selector={TestSelectors.StreamLengthSecondsColumn}>
                    {campaign.streamLengthSeconds}
                  </TableCell>
                  <TableCell data-test-selector={TestSelectors.PlatformsColumn}>
                    {campaign.availablePlatforms}
                  </TableCell>
                  <TableCell data-test-selector={TestSelectors.DetailsColumn}>
                    <ReactMarkdown source={campaign.details} />
                  </TableCell>
                  <TableCell data-test-selector={TestSelectors.GameColumn}>{campaign.game.displayName}</TableCell>
                  <TableCell data-test-selector={TestSelectors.VideoColumn}>{this.renderVideoCell()}</TableCell>
                </TableRow>
              </TableBody>
            </Table>
          </StyledLayout>
          <Layout display={Display.Flex} justifyContent={JustifyContent.Center} padding={{ y: 2 }}>
            <Button data-track-click="bounty-show-campaign-details-close" onClick={onCancel}>
              Close
            </Button>
          </Layout>
        </StyledLayout>
      </StyledLayout>
    );
  }

  renderTitle = (): JSX.Element | undefined => {
    const { campaign } = this.props;
    const title = campaign.title;

    // if override title exist, return it
    if (title) {
      return <CoreText>{title}</CoreText>;
    }

    let displayName = campaign.displayName;

    // if override game displayName exist, use it
    if (!displayName) {
      displayName = campaign.game.displayName;
    }

    const type = campaign.type;
    const streamLengthSeconds = campaign.streamLengthSeconds;

    // if the campaign is game type
    if (type === BountyCampaignType.PAY_TO_STREAM) {
      return (
        <CoreText>
          Stream {displayName} for {streamLengthSeconds / 60} minutes
        </CoreText>
      );
    } else if (type === BountyCampaignType.MISC_TRAILER) {
      // if the campaign is trailer type
      const videos = campaign.video;
      if (videos) {
        // check default video
        const video =
          videos.find((v: BountyCampaignVideo) => {
            // ALL or DEFAULT represent the default value using when showing general campaign details
            return v.country === "ALL" || v.country === "DEFAULT";
          }) || videos[0];

        return (
          <CoreText>
            Watch {video.title} for {streamLengthSeconds / 60} minutes
          </CoreText>
        );
      }
    }
    return (
      <Layout data-test-selector={TestSelectors.ErrorText}>
        <CoreText color={Color.Error}>There is no title for campaign type {campaign.type}</CoreText>
      </Layout>
    );
  };

  renderVideoCell = (): JSX.Element[] => {
    const { campaign } = this.props;
    const videos = campaign.video;
    return (videos || []).map((video, index) => (
      <Layout key={`bounty-row-${video.country}`} display={Display.Block}>
        <CoreLink targetBlank to={video.url}>
          {video.country}
        </CoreLink>
        {index !== (videos || []).length - 1 ? "," : ""}
      </Layout>
    ));
  };
}

export const BountyCampaignDetails: React.ComponentClass<PublicProps> = BountyCampaignDetailsComponent;
