import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { CoreLink, TableCell } from "twitch-core-ui";
import { mockBountyInfo } from "../../mocks";
import { BountyModerationStatusOption } from "../../models";
import { ReviewBountyButton } from "../bounties-table-row-button/component";
import { BountyRowComponent, Props, TestSelectors } from "./component";

Enzyme.configure({ adapter: new Adapter() });

function defaultPropsGenerator(overrides?: Partial<Props>): Props {
  return {
    bounty: mockBountyInfo(1)[0],
    ...overrides
  };
}

const setupShallow = (overrides?: Partial<Props>) => {
  const props = defaultPropsGenerator(overrides);
  return Enzyme.shallow(<BountyRowComponent {...props} />);
};

describe("BountyRowComponent", () => {
  it("renders button properly", () => {
    const wrapper = setupShallow();
    expect(wrapper.find(ReviewBountyButton).length).toBe(2);
  });

  it(`renders the proper cells with pending bounty`, () => {
    const pendingBounty = mockBountyInfo(1)[0];
    const wrapper = setupShallow({ bounty: pendingBounty });

    expect(wrapper.find(TableCell).length).toEqual(10);
    // bountyId
    expect(wrapper.find({ "data-test-selector": TestSelectors.IDColumn }).html()).toContain(pendingBounty.id);
    // stream titles
    expect(wrapper.find({ "data-test-selector": TestSelectors.StreamTitlesColumn }).html()).toContain(
      pendingBounty.streamTitles
    );
    // user Id
    expect(wrapper.find({ "data-test-selector": TestSelectors.UserIdColumn }).html()).toContain(
      pendingBounty.broadcasterInfo.userID
    );
    // language
    expect(wrapper.find({ "data-test-selector": TestSelectors.LanguageColumn }).html()).toContain(
      pendingBounty.broadcasterInfo.language
    );
    // country
    expect(wrapper.find({ "data-test-selector": TestSelectors.CountryColumn }).html()).toContain(
      pendingBounty.broadcasterInfo.country
    );
    // vods
    const vods = pendingBounty.vods;
    if (vods) {
      expect(
        wrapper
          .find({ "data-test-selector": TestSelectors.VodsColumn })
          .find(CoreLink)
          .prop("to")
      ).toBe(vods[0]);
    }
    // approve button
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.ApproveColumn })
        .find(ReviewBountyButton)
        .prop("moderationStatus")
    ).toBe(BountyModerationStatusOption.APPROVE);
    // reject button
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.RejectColumn })
        .find(ReviewBountyButton)
        .prop("moderationStatus")
    ).toBe(BountyModerationStatusOption.REJECT);
  });
});
