import { ReviewBountyButton } from "aegis/features/bounties/components/bounties-table-row-button/component";
import { PortalModal } from "aegis/functionality/components/portal-modal";
import * as React from "react";

import { BountyFailReason, BountyInfo, BountyModerationStatusOption } from "aegis/features/bounties/models";
import {
  Button,
  ButtonType,
  Color,
  CoreLink,
  CoreText,
  Display,
  Layout,
  Select,
  TableCell,
  TableRow
} from "twitch-core-ui";
import { BountyCampaignDetails } from "../bounties-table-row-campaign";
import "./styles.scss";

export enum TestSelectors {
  IDColumn = "bounties-table-row-id-column",
  StreamTitlesColumn = "bounties-table-row-stream-titles-column",
  UserIdColumn = "bounties-table-row-user-id-column",
  CampaignColumn = "bounties-table-row-campaign-column",
  LanguageColumn = "bounties-table-row-language-column",
  CountryColumn = "bounties-table-row-country-column",
  VodColumn = "bounties-table-row-vod-column",
  VodsColumn = "bounties-table-row-vods-column",
  ApproveColumn = "bounties-table-row-approve-column",
  RejectColumn = "bounties-table-row-reject-column",
  ButtonColumn = "bounties-table-row-campaign-button-column",
  ErrorText = "ErrorText"
}

export interface PublicProps {
  bounty: BountyInfo;
}

export type Props = PublicProps;

interface State {
  reason: BountyFailReason | null;
  disabled: boolean;
  campaignVisible: boolean;
}

export class BountyRowComponent extends React.Component<Props, State> {
  public state: State = {
    reason: null,
    disabled: false,
    campaignVisible: false
  };

  public render() {
    const { bounty } = this.props;

    return (
      <TableRow key={`bounty-row-${bounty.id}`}>
        <TableCell data-test-selector={TestSelectors.IDColumn}>{bounty.id}</TableCell>
        <TableCell data-test-selector={TestSelectors.StreamTitlesColumn}>
          <Layout className="bounties-table-row--long">{bounty.streamTitles}</Layout>
        </TableCell>
        <TableCell data-test-selector={TestSelectors.UserIdColumn}>{bounty.broadcasterInfo.userID}</TableCell>
        <TableCell data-test-selector={TestSelectors.LanguageColumn}>{bounty.broadcasterInfo.language}</TableCell>
        <TableCell data-test-selector={TestSelectors.CountryColumn}>{bounty.broadcasterInfo.country}</TableCell>
        <TableCell data-test-selector={TestSelectors.CampaignColumn}>
          {this.renderCampaignModal()}
          <Button
            data-track-click="bounty-show-campaign-details"
            fullWidth
            onClick={this.showCampaign}
            data-test-selector={TestSelectors.ButtonColumn}
          >
            Show Campaign
          </Button>
        </TableCell>
        <TableCell data-test-selector={TestSelectors.VodColumn}>{this.renderVodCell()}</TableCell>
        <TableCell data-test-selector={TestSelectors.VodsColumn}>
          <Layout display={Display.Block}>{this.renderVodsCell()}</Layout>
        </TableCell>
        <TableCell data-test-selector={TestSelectors.ApproveColumn}>
          <ReviewBountyButton
            reason={null}
            bountyID={bounty.id}
            moderationStatus={BountyModerationStatusOption.APPROVE}
            action="Approve"
            type={ButtonType.Success}
            disabled={this.state.disabled}
            onButtonClick={this.onButtonClick}
            onError={this.onError}
          />
        </TableCell>
        <TableCell data-test-selector={TestSelectors.RejectColumn}>
          <Layout className="bounties-table-row--reject-button">
            <Select onChange={this.onChange} defaultValue="">
              <option value="" disabled>
                Choose Reject Reason
              </option>
              {Object.keys(BountyFailReason).map((key: string) => (
                <option key={key} value={key}>
                  {key}
                </option>
              ))}
            </Select>
            <ReviewBountyButton
              disabled={this.state.reason === null || this.state.disabled}
              bountyID={bounty.id}
              reason={this.state.reason}
              moderationStatus={BountyModerationStatusOption.REJECT}
              action="Reject"
              type={ButtonType.Alert}
              onButtonClick={this.onButtonClick}
              onError={this.onError}
            />
          </Layout>
        </TableCell>
      </TableRow>
    );
  }

  onButtonClick = (): void => {
    this.setState({ disabled: true });
  };

  onError = (): void => {
    this.setState({ disabled: false });
  };

  renderVodCell = (): JSX.Element => {
    const { bounty } = this.props;
    const vod = bounty.vod;

    if (!vod) {
      return (
        <Layout data-test-selector={TestSelectors.ErrorText}>
          <CoreText color={Color.Error}>There is no vod associated with this bounty</CoreText>
        </Layout>
      );
    }

    return <iframe width={480} height={270} src={vod.url} allowFullScreen />;
  };

  renderVodsCell = (): JSX.Element[] => {
    const { bounty } = this.props;
    const vods = bounty.vods;
    return (vods || []).map(vod => (
      <CoreLink key={`bounty-row-${bounty.id}-vods-${vod}`} targetBlank to={vod}>
        {vod}
      </CoreLink>
    ));
  };

  onChange = (e: React.FormEvent<HTMLSelectElement>) => {
    const bountyReason = e.currentTarget.value;
    this.setState({ reason: bountyReason as BountyFailReason });
  };

  renderCampaignModal = (): JSX.Element | null => {
    if (!this.state.campaignVisible) {
      return null;
    }
    const { bounty } = this.props;

    return (
      <PortalModal onClose={this.hideCampaign} overrideWidth={true}>
        <BountyCampaignDetails campaign={bounty.campaign} onCancel={this.hideCampaign} />
      </PortalModal>
    );
  };

  private showCampaign = () => {
    this.setState({ campaignVisible: true });
  };

  private hideCampaign = () => {
    this.setState({ campaignVisible: false });
  };
}

export const BountyRow: React.ComponentClass<Props> = BountyRowComponent;
