import * as React from "react";

import gql from "graphql-tag";
import { compose, graphql, GraphqlQueryControls, withApollo } from "react-apollo";

import { BountiesTable } from "aegis/features/bounties/components/bounties-table";
import { PendingBountiesResponse } from "aegis/features/bounties/models";
import { Color, CoreText } from "twitch-core-ui";

export interface PublicProps {
  currentBucket: number;
  bucketCount: number;
  languages: string[];
}

export type Props = PublicProps & GraphQLProps;

interface GraphQLProps {
  data: PendingBountiesResponse & GraphqlQueryControls;
}

export class PendingBountiesContainer extends React.Component<Props, {}> {
  public render() {
    const { pendingBounties, error } = this.props.data;

    if (error) {
      console.error("An error occurred", error);
      return (
        <CoreText color={Color.Error}>
          The following error occurred: {error.message}. Please refresh the page. If this continues to happen, please
          page safety oncall.
        </CoreText>
      );
    }
    return <BountiesTable pendingBounties={pendingBounties || []} />;
  }
}

const PendingBountiesQuery = gql`
  query pendingBounties($bountyBucket: Int, $bucketCount: Int, $languages: [String!]) {
    pendingBounties(bountyBucket: $bountyBucket, bucketCount: $bucketCount, languages: $languages) {
      id
      streamTitles
      vod {
        url
      }
      vods
      broadcasterInfo {
        userID
        country
        language
      }
      campaign {
        title
        sponsor
        type
        streamLengthSeconds
        availablePlatforms
        details
        game {
          displayName
        }
        video {
          title
          country
          url
        }
      }
    }
  }
`;

export const PendingBountiesTable: React.ComponentClass<PublicProps> = compose(
  withApollo,
  graphql<PublicProps, PendingBountiesResponse, {}>(PendingBountiesQuery, {
    options: (props: PublicProps) => {
      return {
        fetchPolicy: "network-only",
        variables: {
          bountyBucket: props.currentBucket,
          bucketCount: props.bucketCount,
          languages: props.languages
        }
      };
    }
  })
)(PendingBountiesContainer);
