import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { FormGroup } from "twitch-core-ui";
import { PendingBountiesTable } from "../bounties-table/container";
import {
  ALL_BUCKET,
  ALL_BUCKET_STATE,
  DEFAULT_BUCKET_COUNT,
  DEFAULT_STATE,
  getBucketState,
  PendingBounties,
  PublicProps
} from "./component";

Enzyme.configure({ adapter: new Adapter() });

const setupShallow = (): Enzyme.ShallowWrapper<PublicProps> => {
  return Enzyme.shallow(<PendingBounties supportedLanguages={[]} />);
};

describe("PendingBounties", () => {
  it("renders correctly", () => {
    const wrapper = setupShallow();
    expect(wrapper.find(FormGroup).find({ label: "Bucket" }).length).toBe(1);
    expect(wrapper.find(PendingBountiesTable).length).toBe(1);
  });

  it("updates state when the Bucket select is changed", () => {
    const wrapper = setupShallow();
    wrapper
      .find(FormGroup)
      .find({ label: "Bucket" })
      .childAt(0)
      .props()
      .onChange({ currentTarget: { value: "2" } });
    expect(wrapper.state("currentBucket")).toBe(2);
    expect(wrapper.state("bucketCount")).toBe(DEFAULT_BUCKET_COUNT);
  });

  it("updates state correctly when the All bucket is selected", () => {
    const wrapper = setupShallow();
    wrapper
      .find(FormGroup)
      .find({ label: "Bucket" })
      .childAt(0)
      .props()
      .onChange({ currentTarget: { value: `${ALL_BUCKET}` } });
    expect(wrapper.state("currentBucket")).toBe(0);
    expect(wrapper.state("bucketCount")).toBe(1);
  });

  describe("getBucketState function", () => {
    it("returns the all bucket state when 'ALL' is passed in", () => {
      const returnVal = getBucketState(ALL_BUCKET);
      expect(returnVal).toEqual(ALL_BUCKET_STATE);
    });

    it("returns state with the proper currentBucket and bucketCount values when a valid bucket number is passed in", () => {
      const returnVal = getBucketState("1");
      expect(returnVal.currentBucket).toEqual(1);
      expect(returnVal.bucketCount).toEqual(DEFAULT_BUCKET_COUNT);
    });

    it("returns the default bucket state if an invalid bucket number is passed in", () => {
      const returnVal = getBucketState("Kappa");
      expect(returnVal.currentBucket).toEqual(DEFAULT_STATE.currentBucket);
      expect(returnVal.bucketCount).toEqual(DEFAULT_STATE.bucketCount);
    });
  });
});
