import * as React from "react";

import * as Enzyme from "enzyme";
import { shallow } from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { createHistory } from "aegis/functionality/utils/contrib/twilight/history";
import { MockDetailedUser } from "aegis/models/mocks";
import { Button, CoreText, Interactable, TextArea, TextType } from "twitch-core-ui";
import { ChatActivityFormComponent, Props, PublicProps, TestSelectors } from "./component";
import { MultiUserInput } from "./multi-user-input";
import { UserPill } from "./multi-user-input/user-pill";

Enzyme.configure({ adapter: new Adapter() });

const defaultPropsGenerator = (overrides?: Partial<PublicProps>): Props => {
  return {
    users: ["user1", "user2", "user3"],
    history: createHistory(),
    data: {
      chatters: [
        MockDetailedUser({ id: "user1" }),
        MockDetailedUser({ id: "user2" }),
        MockDetailedUser({ id: "user3" })
      ],
      activity: []
    },
    ...overrides
  } as Props;
};

const setup = (overrides?: Partial<Props>, empty?: boolean) => {
  const combinedProps: Props = empty ? { users: [], ...overrides } : defaultPropsGenerator(overrides);
  return {
    props: combinedProps,
    wrapper: shallow(<ChatActivityFormComponent {...combinedProps} />)
  };
};

describe("ChatActivityForm", () => {
  it("renders editing state by default", () => {
    const { wrapper } = setup({}, true);

    expect(wrapper.state("editing")).toBe(true);
    expect(wrapper.state("value")).toBe("");
    expect(
      wrapper
        .find(CoreText)
        .find({ type: TextType.H4, bold: true })
        .childAt(0)
        .text()
    ).toEqual("Add user logins or ids to check their channel activity");
    expect(wrapper.find(TextArea)).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelectors.SearchButton })).toHaveLength(1);
  });

  it("renders editing state if no users provided", () => {
    const users: string[] = [];
    const { wrapper } = setup({ users });

    expect(wrapper.state("editing")).toBe(true);
    expect(wrapper.state("value")).toBe(users.join(", "));
  });

  it("renders editing state if error exists", () => {
    const users = ["user1", "user2", "user3"];
    const { wrapper } = setup({ users, errorMessage: "crap" });

    expect(wrapper.state("editing")).toBe(true);
    expect(wrapper.state("value")).toBe(users.join(", "));
  });

  it("renders MultiUserInput with user pills for queried chatters", () => {
    const { wrapper } = setup();

    expect(wrapper.state("editing")).toBe(false);
    expect(wrapper.find(MultiUserInput)).toHaveLength(1);
    expect(
      wrapper
        .find(MultiUserInput)
        .dive()
        .find(UserPill)
    ).toHaveLength(3);
  });

  it("renders the loaded state", () => {
    const { wrapper } = setup();

    expect(wrapper.state("editing")).toBe(false);
    expect(wrapper.find(MultiUserInput)).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelectors.EditButton })).toHaveLength(1);
    expect(wrapper.find({ "data-test-selector": TestSelectors.ClearAllButton })).toHaveLength(1);
    expect(
      wrapper
        .find(CoreText)
        .find({ type: TextType.Span, bold: true })
        .childAt(0)
        .text()
    ).toBe("3 users");
  });

  it("renders empty editing state when 'Clear All' button clicked", () => {
    const { wrapper } = setup();
    expect(wrapper.state("editing")).toBe(false);

    wrapper
      .find({ "data-test-selector": TestSelectors.ClearAllButton })
      .find(Button)
      .simulate("click");
    wrapper.update();

    expect(wrapper.state("editing")).toBe(true);
    expect(wrapper.state("value")).toBe("");
  });

  it("renders sanitized editing state when 'Edit' button clicked", () => {
    const { wrapper } = setup();
    expect(wrapper.state("editing")).toBe(false);

    wrapper
      .find({ "data-test-selector": TestSelectors.EditButton })
      .find(Button)
      .simulate("click");
    wrapper.update();
    expect(wrapper.state("editing")).toBe(true);
    expect(wrapper.state("value")).toBe("user1, user2, user3");
    expect(wrapper.find(TextArea).prop("value")).toBe("user1, user2, user3");
  });

  it("updates the URL when 'Search' is clicked", () => {
    const history = createHistory();
    const { wrapper } = setup({ history }, true);
    expect(wrapper.state("editing")).toBe(true);
    expect(wrapper.setState({ value: "someUserA someUserB" }));
    expect(history.location.search).toBeFalsy();

    wrapper
      .find({ "data-test-selector": TestSelectors.SearchButton })
      .find(Button)
      .simulate("click");

    expect(history.location.search).toBe("?users=someUserA%2CsomeUserB");
  });

  it("updates state and re-renders when a user pill is closed", () => {
    const { wrapper } = setup();

    expect(wrapper.state("value")).toBe("user1, user2, user3");
    wrapper
      .find(MultiUserInput)
      .dive()
      .find(UserPill)
      .at(1)
      .dive()
      .find(Interactable)
      .last()
      .simulate("click");
    wrapper.update();
    expect(wrapper.state("value")).toBe("user1, , user3");
  });
});
