import * as React from "react";

import { History } from "history";
import * as QueryString from "query-string";
import { withRouter } from "react-router";

import { PublicProps as ChatActivityProps } from "src/features/chat-activity";
import {
  AlignItems,
  Button,
  ButtonSize,
  ButtonType,
  Color,
  CoreText,
  Display,
  FontSize,
  Layout,
  LoadingSpinner,
  StyledLayout,
  TextArea,
  TextType
} from "twitch-core-ui";
import { MultiUserInput } from "./multi-user-input";

import "./styles.scss";

export enum TestSelectors {
  SearchButton = "chat-activity-form--search-button",
  EditButton = "chat-activity-form--edit-button",
  ClearAllButton = "chat-activity-form--clear-all-button"
}

export interface PublicProps {
  history?: History;
}

class State {
  value?: string;
  editing: boolean;
}

const deriveFormStateFromProps = (props: Props): State => {
  const { users, errorMessage } = props;
  const noUsers = !users || users.length === 0;

  const derivedState: State = {
    value: users ? users.join(", ") : "",
    editing: !!errorMessage || noUsers
  };
  return derivedState;
};

export type Props = PublicProps & ChatActivityProps;

export class ChatActivityFormComponent extends React.Component<Props, State> {
  constructor(props: Props) {
    super(props);
    this.state = deriveFormStateFromProps(props);
  }

  componentDidUpdate(prevProps: Props) {
    if (prevProps !== this.props) {
      const newState: State = deriveFormStateFromProps(this.props);
      this.setState(newState);
    }
  }

  render() {
    const { users, loading, data } = this.props;
    if (this.state.editing) {
      return (
        <StyledLayout padding={{ top: 1, bottom: 3 }}>
          <Layout padding={{ bottom: 1 }}>
            <CoreText fontSize={FontSize.Size6} type={TextType.H4} bold>
              {"Add user logins or ids to check their channel activity"}
            </CoreText>
            <CoreText fontSize={FontSize.Size7} color={Color.Alt2}>
              {'Use quotes to escape numerical logins. ex: "140078262"'}
            </CoreText>
          </Layout>

          <TextArea
            name="multi-user-input"
            placeholder={'Enter user logins or ids separated with ","'}
            onChange={this.updateValue}
            value={this.state.value}
          />

          <Layout padding={{ y: 2 }}>
            <Button
              data-track-click="chat-activity-search"
              onClick={this.onSubmit}
              data-test-selector={TestSelectors.SearchButton}
            >
              <Layout padding={{ x: 2 }}>{"Search"}</Layout>
            </Button>
          </Layout>
        </StyledLayout>
      );
    }

    return (
      <StyledLayout borderBottom padding={{ y: 0.5 }}>
        <Layout>
          <CoreText type={TextType.Span}>
            <CoreText type={TextType.Span} bold>{`${users ? users.length : 0} users`}</CoreText>
            {" included for channel activity checker"}
          </CoreText>
          <Layout display={Display.Inline} alignItems={AlignItems.Center}>
            <Layout padding={{ left: 1 }} display={Display.Inline}>
              <Button
                data-track-click="chat-activity-edit-users"
                size={ButtonSize.Small}
                type={ButtonType.Hollow}
                data-test-selector={TestSelectors.EditButton}
                onClick={() => {
                  this.setState({ editing: true });
                }}
              >
                {"Edit"}
              </Button>
            </Layout>
            <Layout padding={{ left: 1 }} display={Display.Inline}>
              <CoreText type={TextType.Span}>&bull;</CoreText>
            </Layout>
            <Layout display={Display.Inline}>
              <Button
                data-track-click="chat-activity-clear-all"
                type={ButtonType.Text}
                onClick={this.clearAll}
                size={ButtonSize.Small}
                data-test-selector={TestSelectors.ClearAllButton}
              >
                <StyledLayout color={Color.Error}>{"Clear all"}</StyledLayout>
              </Button>
            </Layout>
          </Layout>
        </Layout>
        {loading ? (
          <LoadingSpinner />
        ) : (
          <MultiUserInput onCloseGenerator={this.closeUser} chatters={data ? data.chatters : []} />
        )}
      </StyledLayout>
    );
  }

  private updateValue = (e: React.FormEvent<HTMLTextAreaElement>) => {
    this.setState({ value: e.currentTarget.value });
  };

  private closeUser = (index: number): (() => void) => {
    const { value } = this.state;
    const { users } = this.props;
    return () => {
      if (!value || !users) {
        return;
      }
      this.setState({ value: value.replace(users[index], "") }, () => {
        this.updateURLParams();
      });
    };
  };

  private clearAll = () => {
    this.setState({ value: "", editing: true }, () => {
      this.updateURLParams();
    });
  };

  private onSubmit = () => {
    this.updateURLParams();
  };

  private updateURLParams() {
    const { value } = this.state;

    const input = value && value.match(/\"?\w+\"?/g);
    const params = QueryString.stringify({ users: input && input.join(",") }, { encode: true, arrayFormat: "none" });
    this.props.history!.push({
      pathname: "/chatactivity",
      search: input ? params : ""
    });
  }
}

export const ChatActivityForm = withRouter(ChatActivityFormComponent);
