import { MessageLog } from "aegis/features/messages";
import { MockMessage } from "aegis/features/messages/mocks";
import { MockUser } from "aegis/models/mocks";
import * as Enzyme from "enzyme";
import { shallow } from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";
import * as React from "react";
import { CoreImage, CoreText } from "twitch-core-ui";
import {
  ChatActivityCard,
  ChatActivityChatColumnNameOverrides,
  ChatActivityChatColumns,
  Props,
  TestSelectors
} from "./component";

Enzyme.configure({ adapter: new Adapter() });

const setup = (overrides?: Partial<Props>) => {
  const props: Props = {
    isOpen: true,
    chatActivityInfo: {
      channel: MockUser({ id: "channel1", login: "channel1login", profileImage: "profileImage.png" }),
      messages: [
        MockMessage({
          body: "herp herp",
          from: MockUser({ id: "user1" })
        }),
        MockMessage({
          body: "derp derp",
          from: MockUser({ id: "user2" })
        })
      ]
    },
    ...overrides
  };

  return {
    props,
    wrapper: shallow(<ChatActivityCard {...props} />)
  };
};

describe("ChatActivityResults", () => {
  it("renders", () => {
    const { wrapper } = setup();

    expect(wrapper.find(CoreImage)).toHaveLength(1);
    expect(wrapper.find(CoreImage).prop("src")).toBe("profileImage.png");
    expect(wrapper.find(MessageLog)).toHaveLength(1);
    expect(
      wrapper
        .find(CoreText)
        .first()
        .childAt(0)
        .text()
    ).toBe("channel1login");
  });

  it("collapses when the header is clicked from expanded state", () => {
    const { wrapper } = setup();
    expect(wrapper.state("isOpen")).toBe(true);

    wrapper.find({ "data-test-selector": TestSelectors.ChannelHeader }).simulate("click");
    wrapper.update();

    expect(wrapper.state("isOpen")).toBe(false);
    expect(wrapper.find(MessageLog)).toHaveLength(0);
  });

  it("expands when the header is clicked from collapsed state", () => {
    const { wrapper } = setup({ isOpen: false });

    expect(wrapper.state("isOpen")).toBe(false);

    wrapper.find({ "data-test-selector": TestSelectors.ChannelHeader }).simulate("click");
    wrapper.update();

    expect(wrapper.state("isOpen")).toBe(true);
    expect(wrapper.find(MessageLog)).toHaveLength(1);
  });

  it("it renders message log in specfic order", () => {
    const { wrapper } = setup();
    expect(wrapper.find(MessageLog).prop("chatColumns")).toBe(ChatActivityChatColumns);
    expect(wrapper.find(MessageLog).prop("columnNameOverride")).toBe(ChatActivityChatColumnNameOverrides);
  });

  it("prop change causes toggle of expansion state", () => {
    const { wrapper } = setup();
    expect(wrapper.state("isOpen")).toBe(true);
    wrapper.setProps({ isOpen: false });
    expect(wrapper.state("isOpen")).toBe(false);
  });
});
