import * as React from "react";

import { MessageColumn, MessageLog } from "aegis/features/messages";
import { errorNotification } from "aegis/functionality/error-notifications";
import { copyTextToClipboard } from "aegis/functionality/utils/contrib/twilight/copy-text-to-clipboard";
import {
  Align,
  AlignItems,
  AlignSelf,
  Aspect,
  Background,
  BorderRadius,
  CoreImage,
  CoreText,
  Display,
  FlexWrap,
  Interactable,
  InteractableType,
  Layout,
  StyledLayout,
  StyledLayoutProps,
  SVG,
  SVGAsset,
  TextType,
  Tooltip,
  TooltipDirection
} from "twitch-core-ui";
import { ChatActivityInfo } from "../../models";

import "./styles.scss";

export enum TestSelectors {
  ChannelHeader = "chat-activity-card--channel-header"
}
export interface Props {
  chatActivityInfo: ChatActivityInfo;
  isOpen: boolean;
}

export interface State {
  isOpen: boolean;
  copyText: string;
}

export const ChatActivityChatColumnNameOverrides = new Map<MessageColumn, string>();
ChatActivityChatColumnNameOverrides.set(MessageColumn.Time, "Time Last Seen");
ChatActivityChatColumnNameOverrides.set(MessageColumn.Message, "Last Chat Sent");
ChatActivityChatColumnNameOverrides.set(MessageColumn.From, "Chatter Username");

export const ChatActivityChatColumns: MessageColumn[] = [
  MessageColumn.From,
  MessageColumn.Message,
  MessageColumn.Time,
  MessageColumn.Actions
];

const defaultCopyText = "Copy all to clipboard";
const imageHeight = 20;

export class ChatActivityCard extends React.Component<Props, State> {
  public state: State = {
    isOpen: this.props.isOpen,
    copyText: defaultCopyText
  };

  public onHeaderClick = () => {
    this.setState(prevState => ({ isOpen: !prevState.isOpen }));
  };

  componentDidUpdate(prevProps: Props) {
    if (prevProps.isOpen !== this.props.isOpen) {
      this.setState({ isOpen: this.props.isOpen });
    }
  }

  render() {
    const { chatActivityInfo } = this.props;
    const { channel } = chatActivityInfo;
    const headerStyles: Partial<StyledLayoutProps> = {
      fullWidth: true,
      borderTop: true,
      borderBottom: !this.state.isOpen,
      background: Background.Alt2,
      display: Display.InlineFlex,
      padding: 0.5,
      alignItems: AlignItems.Center,
      alignSelf: AlignSelf.Stretch
    };
    return (
      <Layout padding={{ y: 1 }}>
        <Layout display={Display.Flex} flexWrap={FlexWrap.NoWrap} alignItems={AlignItems.Stretch}>
          <Layout flexGrow={1}>
            <Interactable
              data-track-click="chat-activity-column-show"
              onClick={this.onHeaderClick}
              type={InteractableType.Overlay}
              data-test-selector={TestSelectors.ChannelHeader}
            >
              <StyledLayout
                {...headerStyles}
                borderLeft
                borderRadius={{
                  topLeft: BorderRadius.Medium,
                  bottomLeft: this.state.isOpen ? BorderRadius.None : BorderRadius.Medium
                }}
              >
                <SVG asset={this.state.isOpen ? SVGAsset.AngleDown : SVGAsset.AngleRight} />
                <StyledLayout
                  className="user-header--profile-image"
                  margin={{ left: 1, right: 0.5 }}
                  borderRadius={BorderRadius.Rounded}
                >
                  <Aspect ratio={1 / 1} align={Align.Center}>
                    <CoreImage
                      src={channel.profileImage}
                      alt={`${channel.login} profile image`}
                      sizes={[{ size: `${imageHeight}px` }]}
                    />
                  </Aspect>
                </StyledLayout>
                <CoreText type={TextType.Span} bold>
                  {channel.login}
                </CoreText>
              </StyledLayout>
            </Interactable>
          </Layout>
          <Layout flexShrink={1} display={Display.InlineFlex} alignItems={AlignItems.Stretch}>
            <Interactable
              onClick={this.copyEntries}
              type={InteractableType.Overlay}
              data-track-click="chat-activity-table-copy"
            >
              <StyledLayout
                {...headerStyles}
                borderRight
                borderRadius={{
                  topRight: BorderRadius.Medium,
                  bottomRight: this.state.isOpen ? BorderRadius.None : BorderRadius.Medium
                }}
              >
                <Tooltip direction={TooltipDirection.Left} label={this.state.copyText}>
                  <Layout display={Display.Flex} alignItems={AlignItems.Center}>
                    <CoreText type={TextType.Span}>Copy All</CoreText>
                    <Layout margin={{ left: 0.5 }}>
                      <SVG asset={SVGAsset.Copy} height={imageHeight} />
                    </Layout>
                  </Layout>
                </Tooltip>
              </StyledLayout>
            </Interactable>
          </Layout>
        </Layout>

        {this.state.isOpen && (
          <MessageLog
            className="chat-activity-card--message-log"
            entries={chatActivityInfo.messages}
            columnNameOverride={ChatActivityChatColumnNameOverrides}
            chatColumns={ChatActivityChatColumns}
          />
        )}
      </Layout>
    );
  }

  private copyEntries = () => {
    const { chatActivityInfo } = this.props;
    const text = chatActivityInfo.messages
      .map(
        m =>
          `#${chatActivityInfo.channel.login}   ${m.from && m.from.login}${
            m.badges.some(b => b.title === "Moderator") ? " [MOD]" : ""
          }`
      )
      .join("\n");
    if (copyTextToClipboard(text)) {
      this.setState({
        copyText: "Copied!"
      });
      setTimeout(() => {
        this.setState({
          copyText: defaultCopyText
        });
      }, 2000);
    } else {
      errorNotification("There was an error copying the array!");
    }
  };
}
