import * as React from "react";

import { shallow } from "enzyme";
import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { MockMessage } from "aegis/features/messages/mocks";
import "aegis/functionality/date-extensions.ts";
import { MockUser } from "aegis/models/mocks";
import { Button } from "twitch-core-ui";
import { ChatActivityCard } from "./chat-activity-card";
import { ChatActivityResults, Props, TestSelectors } from "./component";

Enzyme.configure({ adapter: new Adapter() });

const setup = (overrides?: Partial<Props>) => {
  const props: Props = {
    users: ["users1", "users2", "users3"],
    activity: [
      {
        channel: MockUser({ id: "channel1" }),
        messages: [MockMessage({ id: "msg1" })]
      },
      {
        channel: MockUser({ id: "channel2" }),
        messages: [MockMessage({ id: "msg2" })]
      }
    ],
    ...overrides
  };

  return {
    props,
    wrapper: shallow(<ChatActivityResults {...props} />)
  };
};

describe("ChatActivityResults", () => {
  it("renders", () => {
    const { wrapper, props } = setup();

    const numberOfCards = props.activity.length;
    expect(wrapper.find(ChatActivityCard).length).toBe(numberOfCards);
  });

  it("clicking 'Collapse All' toggles to 'Expand All'", () => {
    const { wrapper } = setup();
    const toggleButton = wrapper.find({ "data-test-selector": TestSelectors.ToggleAllButton }).find(Button);

    expect(wrapper.state("allOpen")).toBe(true);
    expect(toggleButton.childAt(0).text()).toBe("Collapse All");
    toggleButton.simulate("click");
    wrapper.update();

    expect(wrapper.state("allOpen")).toBe(false);
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.ToggleAllButton })
        .find(Button)
        .childAt(0)
        .text()
    ).toBe("Expand All");
  });

  it("clicking 'Expand All' toggles to 'Collapse All'", () => {
    const { wrapper } = setup();
    wrapper.setState({ allOpen: false });
    wrapper.update();
    const toggleButton = wrapper.find({ "data-test-selector": TestSelectors.ToggleAllButton }).find(Button);

    expect(toggleButton.childAt(0).text()).toBe("Expand All");
    toggleButton.simulate("click");
    wrapper.update();

    expect(wrapper.state("allOpen")).toBe(true);
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.ToggleAllButton })
        .find(Button)
        .childAt(0)
        .text()
    ).toBe("Collapse All");
  });

  it("displays number of channels from results", () => {
    const { wrapper, props } = setup();

    const numberOfCards = props.activity.length;
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.NumberOfChannels })
        .childAt(0)
        .text()
    ).toBe(`${numberOfCards} Channels`);
  });

  it("renders placeholder text for no input", () => {
    const { wrapper } = setup({ users: null });

    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.PlaceholderHint })
        .childAt(0)
        .text()
    ).toBe("Please add usernames first from above.");
  });
});
