import * as React from "react";

import * as Fragments from "aegis/fragments";
import gql from "graphql-tag";

import { Query } from "aegis/functionality/apollo-wrapper";
import { errorNotification } from "aegis/functionality/error-notifications";
import { ApolloError } from "apollo-client";
import { Color, CoreText, LoadingSpinner, TextType } from "twitch-core-ui";
import { ChatActivityComponent, PublicProps } from "./component";
import { ChatActivityData } from "./models";

export const FETCH_CHAT_ACTIVITY = gql`
  query chatActivity($users: [String!]) {
    chatActivity(users: $users) {
      chatters {
        ...UserFields
        ...DetailedUserFields
      }
      activity {
        channel {
          profileImage
          login
        }
        messages {
          ...MessageFields
        }
      }
    }
  }

  ${Fragments.MessageFields}
  ${Fragments.UserFields}
  ${Fragments.DetailedUserFields}
`;

export interface Variables {
  users: string[] | null;
}

export interface ChatActivityResponse {
  chatActivity: ChatActivityData;
}

export class ChatActivityQuery extends Query<ChatActivityResponse, Variables> {}

export class ChatActivity extends React.Component<PublicProps> {
  render() {
    const { users } = this.props;
    const variables: Variables = { users };

    if (!users || users.length === 0) {
      return <ChatActivityComponent {...this.props} />;
    }

    return (
      <ChatActivityQuery
        displayName="ChatActivity"
        query={FETCH_CHAT_ACTIVITY}
        variables={variables}
        fetchPolicy="network-only"
        errorPolicy="none" // onError handler doesn't fire with other error policies https://github.com/apollographql/react-apollo/issues/2590
        onError={(e: ApolloError) => {
          errorNotification("Failed to fetch chat activity", {
            error: `${e.message}\n${location.href}`,
            copyErrorToClipboard: true
          });
        }}
      >
        {({ loading, error, data }) => {
          const errorMessage = (error && !loading && `Failed to fetch chat activity: ${error.message}`) || "";
          let results: ChatActivityData | undefined;
          if (!loading && !error) {
            results = data && data.chatActivity;
          }

          return (
            <ChatActivityComponent loading={loading} errorMessage={errorMessage} data={results} {...this.props}>
              {loading && <LoadingSpinner />}
              {error && (
                <CoreText type={TextType.Strong} color={Color.Error}>
                  {errorMessage}
                </CoreText>
              )}
            </ChatActivityComponent>
          );
        }}
      </ChatActivityQuery>
    );
  }
}
