import * as React from "react";

import { ChatColumnNameOverrides, formatMessageEntry, Message, MessageLog } from "aegis/features/messages";
import { StrikeButton } from "aegis/features/strike-button";
import { TranslateButton } from "aegis/features/translate";
import { VodPopoutButton } from "aegis/features/vod-popout-button";
import { User } from "aegis/models";
import { Button, ButtonState, ButtonType } from "twitch-core-ui";

export interface Props {
  chatEntries: Message[];
  nextPage?: () => void;
  prevPage?: () => void;
  loading?: boolean;
}

export enum TestSelectors {
  MessageLog = "chat-log-message-log"
}

export class ChatLogsComponent extends React.Component<Props> {
  public render() {
    const { prevPage, nextPage, loading, chatEntries } = this.props;
    const strikeableUser = this.getUserIfOneSender();
    return (
      <div>
        <MessageLog
          entries={chatEntries}
          columnNameOverride={ChatColumnNameOverrides}
          data-test-selector={TestSelectors.MessageLog}
          additionalActionFactory={this.additionalActionsFactory}
        />
        <Button
          data-track-click="chatlogs-previous-page"
          onClick={prevPage}
          type={ButtonType.Hollow}
          disabled={!prevPage || loading}
          state={loading ? ButtonState.Loading : ButtonState.Default}
        >
          Previous Page
        </Button>
        <Button
          data-track-click="chatlogs-next-page"
          onClick={nextPage}
          type={ButtonType.Hollow}
          disabled={!nextPage || loading}
          state={loading ? ButtonState.Loading : ButtonState.Default}
        >
          Next Page
        </Button>
        <TranslateButton
          type={ButtonType.Hollow}
          label="Translate All Rows"
          for={this.props.chatEntries
            .map(
              message =>
                `#${message.target && message.target.displayName} - ${message.from && message.from.displayName}: ${
                  message.body
                }`
            )
            .join("\n")}
        />
        <StrikeButton
          type={ButtonType.Hollow}
          user={strikeableUser}
          banFormProps={{
            targetUserID: strikeableUser && strikeableUser.id,
            targetUser: strikeableUser && strikeableUser.login,
            defaultContent: "chat",
            placeholders: {
              chatlog: this.props.chatEntries.map(message => formatMessageEntry(message)).join("\n")
            }
          }}
          label="Strike All Rows"
        />
      </div>
    );
  }

  private additionalActionsFactory = (message: Message) => {
    if (!message.target) {
      return [];
    }

    const timestamp = parseInt(message.sentTimestamp, 10) / 1000; // ms to seconds

    return [
      <VodPopoutButton
        key={message.id}
        tooltip={"Go to VOD at the time of the message"}
        channelID={message.target.id}
        timestamp={timestamp}
      />
    ];
  };

  // Checks user for all chatEntries, if all are from the same user it returns that user
  // Used for "Strike All" button
  private getUserIfOneSender = (): User | undefined => {
    const users = new Map();
    this.props.chatEntries.forEach(message => message.from && users.set(message.from.id, message.from));
    if (users.size !== 1) {
      return;
    }
    return Array.from(users.values())[0];
  };
}
