import * as React from "react";

import gql from "graphql-tag";

import { Message } from "aegis/features/messages";
import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { PagedResponse } from "aegis/functionality/models";
import { ChatStore } from "aegis/stores";
import { inject, observer } from "mobx-react";
import { LoadingFill } from "../loading-fill";
import { ChatLogsComponent } from "./component";

export const FETCH_CHAT_LOGS_QUERY = gql`
  query chatentries(
    $user: String
    $channel: String
    $first: Int
    $after: Cursor
    $timeStart: String
    $timeEnd: String!
    $regex: String
  ) {
    chatentries(
      user: $user
      channel: $channel
      first: $first
      after: $after
      timeStart: $timeStart
      timeEnd: $timeEnd
      regex: $regex
    ) {
      edges {
        node {
          ...MessageFields
        }
      }
      pageInfo {
        endCursor
      }
    }
  }

  ${Fragments.UserFields}
  ${Fragments.MessageFields}
`;

export interface ChatLogsInput {
  user?: string;
  channel?: string;
  first: number;
  timeStart: string;
  timeEnd: string;
  regex?: string | null;
  after?: string;
  before?: string;
}

export type ChatLogsResponse = {
  chatentries: PagedResponse<Message>;
};

export class ChatLogsQuery extends Query<ChatLogsResponse, ChatLogsInput> {}

export class ChatLogsContainer extends React.Component<ChatLogsInput> {
  private previousPageCursors: (string | undefined)[] = [];
  private currentPageCursor: string | undefined = this.props.after;

  public render() {
    if (!this.props.user && !this.props.channel) {
      return null;
    }
    return (
      <ChatLogsQuery
        displayName="ChatLogsContainer"
        query={FETCH_CHAT_LOGS_QUERY}
        variables={{
          ...this.props
        }}
      >
        {({ loading, error, data, fetchMore }) => {
          if (loading) {
            return <LoadingFill />;
          }
          if (error || !data || !data.chatentries) {
            console.error("Error is:", error);
            console.error("Data is:", data);
            return <div>Error</div>;
          }
          const chatEntries: Message[] = data.chatentries.edges.map(edge => edge.node);
          const pageInfo = data.chatentries.pageInfo;

          const loadPage = (next: boolean, cursor: string | undefined) => () => {
            if (!next) {
              this.previousPageCursors.pop();
            }

            fetchMore({
              query: FETCH_CHAT_LOGS_QUERY,
              variables: {
                ...this.props,
                after: cursor
              },
              updateQuery: (previousResult: ChatLogsResponse, { fetchMoreResult }) => {
                if (!fetchMoreResult) {
                  return previousResult;
                }
                if (next) {
                  this.previousPageCursors.push(this.currentPageCursor);
                }
                this.currentPageCursor = cursor;
                return fetchMoreResult;
              }
            });
          };

          const prevPage = this.currentPageCursor
            ? loadPage(false, this.previousPageCursors[this.previousPageCursors.length - 1])
            : undefined;
          const nextPage = pageInfo.endCursor ? loadPage(true, pageInfo.endCursor) : undefined;
          return (
            <ChatLogsComponent chatEntries={chatEntries} nextPage={nextPage} prevPage={prevPage} loading={loading} />
          );
        }}
      </ChatLogsQuery>
    );
  }
}

@inject("chatStore")
@observer
export class ChatLogs extends React.Component<{ chatStore?: ChatStore }> {
  public render() {
    if (!this.props.chatStore) {
      return <div>ERROR</div>;
    }
    return <ChatLogsContainer {...this.props.chatStore.paramsAsInput} />;
  }
}
