import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";

import { DeleteChatroomButton } from "aegis/features/chatrooms/delete-chatroom-button";
import { UpdateChatroomButton } from "aegis/features/chatrooms/update-chatroom-button";
import { Chatroom as ChatroomModel } from "aegis/models";
import {
  Background,
  Button,
  ButtonType,
  Card,
  CardBody,
  CoreText,
  Display,
  FlexDirection,
  FormGroup,
  Input,
  InputType,
  JustifyContent,
  Layout
} from "twitch-core-ui";

export interface Props {
  room: ChatroomModel;
}

export enum TestSelectors {
  ID = "chatroom-id",
  Name = "chatroom-name",
  Topic = "chatroom-topic",
  OwnerID = "chatroom-owner-id",
  OwnerType = "chatroom-owner-type",
  MinAllowRole = "chatroom-min-allow-role",
  Modes = "chatroom-modes",
  UpdateButton = "chatroom-update-button",
  EditButton = "chatroom-edit-button",
  DeleteButton = "chatroom-delete-button"
}

@observer
export class Chatroom extends React.Component<Props> {
  @observable private name = this.props.room.name;
  @observable private topic = this.props.room.topic;
  @observable private viewOnly = true;

  public render() {
    const { room } = this.props;

    const nameComponent = this.viewOnly ? (
      <CoreText data-test-selector={TestSelectors.Name}>{this.name}</CoreText>
    ) : (
      <Input
        type={InputType.Text}
        value={this.name}
        onChange={this.updateName}
        data-test-selector={TestSelectors.Name}
      />
    );

    const topicComponent = this.viewOnly ? (
      <CoreText data-test-selector={TestSelectors.Topic}>{this.topic}</CoreText>
    ) : (
      <Input
        type={InputType.Text}
        value={this.topic}
        onChange={this.updateTopic}
        data-test-selector={TestSelectors.Topic}
      />
    );

    const updateRoomButton = this.viewOnly ? (
      <Button
        data-track-click="rooms-embed-edit"
        type={ButtonType.Hollow}
        onClick={this.enableUpdate}
        data-test-selector={TestSelectors.EditButton}
      >
        Edit
      </Button>
    ) : (
      <UpdateChatroomButton
        id={room.id}
        name={this.name}
        topic={this.topic}
        data-test-selector={TestSelectors.UpdateButton}
      />
    );

    return (
      <Card elevation={2} background={Background.Alt}>
        <CardBody>
          <Layout display={Display.Flex} flexDirection={FlexDirection.Row} justifyContent={JustifyContent.Center}>
            <FormGroup label="ID">
              <CoreText data-test-selector={TestSelectors.ID}>{room.id}</CoreText>
            </FormGroup>
            <FormGroup label="Name">{nameComponent}</FormGroup>
            <FormGroup label="Topic">{topicComponent}</FormGroup>
            <FormGroup label="Owner ID">
              <CoreText data-test-selector={TestSelectors.OwnerID}>{room.ownerID}</CoreText>
            </FormGroup>
            <FormGroup label="Owner Type">
              <CoreText data-test-selector={TestSelectors.OwnerType}>{room.ownerType}</CoreText>
            </FormGroup>
            <FormGroup label="Minimum Allowed Role">
              <CoreText data-test-selector={TestSelectors.MinAllowRole}>{room.minimumAllowedRole}</CoreText>
            </FormGroup>
            <FormGroup label="Modes">
              <CoreText data-test-selector={TestSelectors.Modes}>{room.modes}</CoreText>
            </FormGroup>
            <FormGroup label="Update">{updateRoomButton}</FormGroup>
            <FormGroup label="Delete">
              <DeleteChatroomButton id={room.id} data-test-selector={TestSelectors.DeleteButton} />
            </FormGroup>
          </Layout>
        </CardBody>
      </Card>
    );
  }

  @action
  private updateName: React.FormEventHandler<HTMLInputElement> = e => {
    this.name = e.currentTarget.value;
  };

  @action
  private updateTopic: React.FormEventHandler<HTMLInputElement> = e => {
    this.topic = e.currentTarget.value;
  };

  @action
  private enableUpdate: React.MouseEventHandler<HTMLBodyElement> = _ => {
    this.viewOnly = false;
  };
}
