import * as React from "react";

import gql from "graphql-tag";
import { action, observable } from "mobx";
import { observer } from "mobx-react";
import { Mutation } from "react-apollo";

import { WithConfirmationModal } from "aegis/functionality/components/confirmation-modal";
import { DeleteChatroomButtonComponent } from "./component";

export interface Props {
  id: string;
}

export type DeleteChatroomResponse = {
  deleteChatroom: {
    success: boolean;
  };
};

export type DeleteChatroomInput = {
  id: string;
};

export const DELETE_ROOM = gql`
  mutation deleteChatroom($id: ID!) {
    deleteChatroom(input: { id: $id }) {
      success
    }
  }
`;

export class DeleteChatroomMutation extends Mutation<DeleteChatroomResponse, DeleteChatroomInput> {}

@observer
export class DeleteChatroomButton extends React.Component<Props> {
  @observable private deleted: boolean = false;
  @observable private error: Error | undefined;

  public render() {
    const { id } = this.props;
    const error = this.error;
    const deleted = this.deleted;

    return (
      <DeleteChatroomMutation mutation={DELETE_ROOM} onError={this.onError} onCompleted={this.onCompleted}>
        {(deleteChatroom, { loading }) => {
          return (
            <WithConfirmationModal
              title="Delete Chatroom"
              body="Are you sure you want to delete this chatroom?"
              confirmLabelOverride="Delete"
              onConfirm={() => {
                deleteChatroom({ variables: { id } });
              }}
            >
              {confirm => {
                return (
                  <DeleteChatroomButtonComponent onClick={confirm} error={error} deleted={deleted} loading={loading} />
                );
              }}
            </WithConfirmationModal>
          );
        }}
      </DeleteChatroomMutation>
    );
  }

  @action
  private onError = (error: Error) => {
    console.error("Failed to delete chatroom: ", error);
    this.error = error;
  };

  @action
  private onCompleted = (response: DeleteChatroomResponse) => {
    this.deleted = response.deleteChatroom.success;
  };
}
