import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { MockMessage } from "aegis/features/messages/mocks";
import { fakeMessage } from "aegis/features/messages/mocks";
import { ConfirmationModalConfirmSelector } from "aegis/functionality/components/confirmation-modal/component";
import { mockErrorResponse, mockSuccessResponse } from "aegis/tests/apollo";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { waitSomeMs } from "aegis/tests/utils/wait-some-ms";
import { Button } from "twitch-core-ui";
import { DELETE_MESSAGE, DeleteChatroomMessageButton, Props } from ".";
import { TestSelectors } from "./component";

Enzyme.configure({ adapter: new Adapter() });

const successInput = {
  roomID: "successRoom",
  messageID: "successMessage"
};
const errorInput = {
  roomID: "errorRoom",
  messageID: "errorMessage"
};

const mockDeletionSuccess = mockSuccessResponse(DELETE_MESSAGE, successInput, {
  data: {
    deleteChatroomMessage: {
      success: true,
      __typename: "DeleteChatroomMessagePayload"
    }
  }
});

const mockDeletionError = mockErrorResponse(DELETE_MESSAGE, errorInput);

const mocks = [mockDeletionSuccess, mockDeletionError];

function defaultPropsGenerator(overrides?: Partial<Props>): Props {
  return { roomID: "", message: fakeMessage, ...overrides };
}

function setupMount(overrides?: Partial<Props>) {
  const props = defaultPropsGenerator();
  return Enzyme.mount(
    wrapWithGQLProvider(<DeleteChatroomMessageButton {...props} {...overrides} />, mocks).wrappedComponent
  );
}

describe("DeleteChatroomMessageButton", () => {
  const modalRoot = document.createElement("div");
  modalRoot.setAttribute("id", "portal-modal-root");
  const body = document.querySelector("body");
  if (body) {
    body.appendChild(modalRoot);
  }

  const globalDiv = document.createElement("div");
  document.body.appendChild(globalDiv);

  it("renders", () => {
    const wrapper = setupMount({
      message: MockMessage()
    });
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(false);
  });

  it("disables button when loading gql mutation query", () => {
    const wrapper = setupMount({
      roomID: successInput.roomID,
      message: MockMessage({
        id: successInput.messageID
      })
    });
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(false);
    wrapper
      .find({ "data-test-selector": TestSelectors.Button })
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(true);
  });

  it("disables button if message is successfully deleted", async () => {
    const wrapper = setupMount({
      roomID: successInput.roomID,
      message: MockMessage({
        id: successInput.messageID
      })
    });
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(false);
    wrapper
      .find({ "data-test-selector": TestSelectors.Button })
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");
    await waitSomeMs();
    wrapper.update();
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(true);
  });

  it("disables button when if message is already deleted", () => {
    const wrapper = setupMount({
      message: MockMessage({
        isDeleted: true
      })
    });
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(true);
  });

  it("does not disables button if message deletion is unsuccessful", async () => {
    spyOn(console, "error"); // console error is expected

    const wrapper = setupMount({
      roomID: errorInput.roomID,
      message: MockMessage({
        id: errorInput.messageID
      })
    });
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(false);
    wrapper
      .find({ "data-test-selector": TestSelectors.Button })
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");
    await waitSomeMs();
    wrapper.update();
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(false);
  });
});
