import * as React from "react";

import gql from "graphql-tag";
import { action, observable } from "mobx";
import { observer } from "mobx-react";
import { Mutation } from "react-apollo";

import { Message } from "aegis/features/messages";
import { WithConfirmationModal } from "aegis/functionality/components/confirmation-modal";
import { DeleteChatroomMessageButtonComponent } from "./component";

export interface Props {
  message: Message;
  roomID: string;
}

export type DeleteChatroomMessageResponse = {
  deleteChatroomMessage: {
    success: boolean;
  };
};

export type DeleteChatroomMessageInput = {
  roomID: string;
  messageID?: string;
};

export const DELETE_MESSAGE = gql`
  mutation deleteChatroomMessage($roomID: ID!, $messageID: ID!) {
    deleteChatroomMessage(input: { roomID: $roomID, messageID: $messageID }) {
      success
    }
  }
`;

export class DeleteChatroomMessageMutation extends Mutation<
  DeleteChatroomMessageResponse,
  DeleteChatroomMessageInput
> {}

@observer
export class DeleteChatroomMessageButton extends React.Component<Props> {
  @observable private deleted: boolean = false;
  @observable private error: Error | undefined;

  public render() {
    const { message, roomID } = this.props;
    const error = this.error;
    const deleted = this.deleted;

    return (
      <DeleteChatroomMessageMutation mutation={DELETE_MESSAGE} onError={this.onError} onCompleted={this.onCompleted}>
        {(deleteChatroomMessage, { loading }) => {
          return (
            <WithConfirmationModal
              title="Delete Chatroom Message"
              body="Are you sure you want to delete this message from this chatroom?"
              confirmLabelOverride="Delete"
              onConfirm={() => {
                deleteChatroomMessage({ variables: { roomID, messageID: message.id } });
              }}
            >
              {confirm => {
                return (
                  <DeleteChatroomMessageButtonComponent
                    onClick={confirm}
                    loading={loading}
                    deleted={message.isDeleted || deleted}
                    error={error}
                  />
                );
              }}
            </WithConfirmationModal>
          );
        }}
      </DeleteChatroomMessageMutation>
    );
  }

  @action
  private onError = (error: Error) => {
    console.error("Failed to delete chatroom message: ", error);
    this.error = error;
  };

  @action
  private onCompleted = (response: DeleteChatroomMessageResponse) => {
    this.deleted = response.deleteChatroomMessage.success;
  };
}
