import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";
import { MockedProvider } from "react-apollo/test-utils";

import { MockChatroom } from "aegis/models/mocks";
import { mockErrorResponse, mockSuccessResponse } from "aegis/tests/apollo";
import { waitSomeMs } from "aegis/tests/utils/wait-some-ms";
import { Button } from "twitch-core-ui";
import { Props, UPDATE_ROOM, UpdateChatroomButton } from ".";

Enzyme.configure({ adapter: new Adapter() });

const successInput = {
  id: "success",
  name: "name",
  topic: "topic"
};

const errorInput = {
  id: "error",
  name: "name",
  topic: "topic"
};

const mockUpdateSuccess = mockSuccessResponse(UPDATE_ROOM, successInput, {
  data: {
    updateChatroom: {
      room: MockChatroom(),
      __typename: "UpdateChatroomPayload"
    }
  }
});

const mockUpdateError = mockErrorResponse(UPDATE_ROOM, errorInput);

const mocks = [mockUpdateSuccess, mockUpdateError];

function defaultPropsGenerator(overrides?: Partial<Props>): Props {
  return {
    id: "id",
    name: "name",
    topic: "topic",
    ...overrides
  };
}

function setupMount(overrides?: Partial<Props>) {
  const props = defaultPropsGenerator();
  return Enzyme.mount(
    <MockedProvider mocks={mocks}>
      <UpdateChatroomButton {...props} {...overrides} />
    </MockedProvider>
  );
}

describe("Update chatroom button", () => {
  it("renders", () => {
    const wrapper = setupMount();
    expect(wrapper.find(Button).prop("disabled")).toBe(false);
  });

  it("disables button when loading gql mutation query", () => {
    const wrapper = setupMount({
      id: successInput.id
    });
    expect(wrapper.find(Button).prop("disabled")).toBe(false);
    wrapper.find(Button).simulate("click");
    expect(wrapper.find(Button).prop("disabled")).toBe(true);
  });

  it("does not disables button if room is successfully updated", async () => {
    const wrapper = setupMount({
      id: successInput.id
    });
    expect(wrapper.find(Button).prop("disabled")).toBe(false);
    wrapper.find(Button).simulate("click");
    await waitSomeMs();
    wrapper.update();
    expect(wrapper.find(Button).prop("disabled")).toBe(false);
  });

  it("does not disables button if update is unsuccessful", async () => {
    spyOn(console, "error"); // console error is expected

    const wrapper = setupMount({
      id: errorInput.id
    });
    expect(wrapper.find(Button).prop("disabled")).toBe(false);
    wrapper.find(Button).simulate("click");
    await waitSomeMs();
    wrapper.update();
    expect(wrapper.find(Button).prop("disabled")).toBe(false);
  });
});
