import * as React from "react";

import gql from "graphql-tag";
import { action, observable } from "mobx";
import { observer } from "mobx-react";
import { Mutation } from "react-apollo";

import { ChatroomFields } from "aegis/fragments";
import { UpdateChatroomButtonComponent } from "./component";

export interface Props {
  id: string;
  name: string;
  topic: string;
}

export type UpdateChatroomResponse = {
  updateChatroom: {
    success: boolean;
  };
};

export type UpdateChatroomInput = {
  id: string;
  name: string;
  topic: string;
};

export const UPDATE_ROOM = gql`
  mutation updateChatroom($id: ID!, $name: String!, $topic: String!) {
    updateChatroom(input: { id: $id, name: $name, topic: $topic }) {
      room {
        ...ChatroomFields
      }
    }
  }
  ${ChatroomFields}
`;

export class UpdateChatroomMutation extends Mutation<UpdateChatroomResponse, UpdateChatroomInput> {}

@observer
export class UpdateChatroomButton extends React.Component<Props> {
  @observable private success: boolean = false;
  @observable private error: Error | undefined;

  public render() {
    const { id, name, topic } = this.props;
    const error = this.error;
    const success = this.success;

    return (
      <UpdateChatroomMutation mutation={UPDATE_ROOM} onError={this.onError} onCompleted={this.onComplete}>
        {(updateChatroom, { loading }) => {
          return (
            <UpdateChatroomButtonComponent
              onClick={() => {
                updateChatroom({ variables: { id: id, name: name, topic: topic } });
              }}
              loading={loading}
              success={success}
              error={error}
            />
          );
        }}
      </UpdateChatroomMutation>
    );
  }

  @action
  private onError = (error: Error) => {
    console.error("Failed to update chatroom: ", error);
    this.error = error;
  };

  @action
  private resetSuccess = () => {
    this.success = false;
  };

  @action
  private onComplete = () => {
    this.success = true;
    setTimeout(this.resetSuccess, 3000); // So user can click this again
  };
}
