import * as React from "react";

import * as classnames from "classnames";
import { History } from "history";
import { observable } from "mobx";
import { inject, observer } from "mobx-react";
import * as QueryString from "query-string";
import { withRouter } from "react-router";

import { removeTZOffset, toDateTimeLocal } from "aegis/functionality/utils/timestamp-conversions";
import { ChatStore, ChatStoreParams } from "aegis/stores";
import {
  Background,
  Button,
  Column,
  FormGroup,
  Grid,
  Input,
  InputType,
  Layout,
  Position,
  StyledLayout
} from "twitch-core-ui";

import "./styles.scss";

export const DEFAULT_SEARCH_STATE: ChatStoreParams = {
  after: "0",
  before: Date.now().toString(),
  channelids: "",
  userids: "",
  regex: ""
};

interface Props {
  chatStore?: ChatStore;
  history?: History;
}

@inject("chatStore")
@observer
class ChatSearchComponent extends React.Component<Props> {
  @observable private searchState: ChatStoreParams = { ...DEFAULT_SEARCH_STATE, ...this.props.chatStore!.params };
  private inputActive = false;
  @observable private invalidTimeWindow = false;

  public componentWillMount() {
    const chatStore = this.props.chatStore;
    this.searchState = chatStore!.params;
    document.addEventListener("keyup", this.enterHandler);
  }

  public componentWillUnmount() {
    this.searchState = DEFAULT_SEARCH_STATE;
    this.props.chatStore!.update(DEFAULT_SEARCH_STATE);
    document.removeEventListener("keyup", this.enterHandler);
  }

  public render() {
    const params = this.props.chatStore ? this.props.chatStore.params : DEFAULT_SEARCH_STATE;

    const dtAfter = toDateTimeLocal(params.after, 0);
    const dtBefore = toDateTimeLocal(params.before, Date.now());
    const focusHandlers = {
      onFocus: () => (this.inputActive = true),
      onBlur: () => (this.inputActive = false)
    };
    this.inputActive = false;

    // These classes are pulled from CoreUI to style the dateTime inputs
    // to match the CoreUI <Input> component
    const dtClasses = classnames(
      "tw-block",
      "tw-border-radius-medium",
      "tw-font-size-6",
      "tw-full-width",
      "tw-pd-l-1",
      "tw-pd-r-1",
      "tw-pd-y-05",
      "tw-input",
      {
        "tw-input--error": this.invalidTimeWindow
      }
    );
    const searchHelp = "This can be comma-separated for multiple IDs and logins.";

    return (
      <StyledLayout className="chatlog-search" background={Background.Alt} border>
        <Grid>
          <Column cols={4}>
            <Layout padding={1}>
              <FormGroup label="Users" hint={searchHelp}>
                <Input
                  key={params.userids}
                  type={InputType.Text}
                  placeholder="17089325"
                  autoComplete="on"
                  onChange={(e: React.FormEvent<HTMLInputElement>): void => {
                    this.searchState.userids = e.currentTarget.value;
                  }}
                  defaultValue={params.userids}
                  {...focusHandlers}
                />
              </FormGroup>
            </Layout>
          </Column>
          <Column cols={4}>
            <Layout padding={1}>
              <FormGroup label="Channels" hint={searchHelp}>
                <Input
                  key={params.channelids}
                  type={InputType.Text}
                  placeholder="17089325"
                  autoComplete="on"
                  onChange={(e: React.FormEvent<HTMLInputElement>): void => {
                    this.searchState.channelids = e.currentTarget.value;
                  }}
                  defaultValue={params.channelids}
                  {...focusHandlers}
                />
              </FormGroup>
            </Layout>
          </Column>
          <Column cols={4}>
            <Layout padding={1}>
              <FormGroup label="Message Filter">
                <Input
                  key={params.regex}
                  type={InputType.Text}
                  onChange={(e: React.FormEvent<HTMLInputElement>): void => {
                    this.searchState.regex = e.currentTarget.value;
                  }}
                  defaultValue={params.regex}
                  {...focusHandlers}
                />
              </FormGroup>
            </Layout>
          </Column>
          <Column cols={4}>
            <Layout padding={1}>
              <FormGroup label="Start Time">
                <input
                  key={params.after}
                  className={dtClasses}
                  type="datetime-local"
                  onChange={(e: React.FormEvent<HTMLInputElement>): void => {
                    const newVal = removeTZOffset(Date.parse(e.currentTarget.value)).toString();
                    if (newVal < this.searchState.before) {
                      this.invalidTimeWindow = false;
                      this.searchState.after = newVal;
                    } else {
                      this.invalidTimeWindow = true;
                    }
                  }}
                  defaultValue={dtAfter}
                  {...focusHandlers}
                />
              </FormGroup>
            </Layout>
          </Column>
          <Column cols={4}>
            <Layout padding={1}>
              <FormGroup label="End Time">
                <input
                  key={params.before}
                  className={dtClasses}
                  type="datetime-local"
                  onChange={(e: React.FormEvent<HTMLInputElement>): void => {
                    const newVal = removeTZOffset(Date.parse(e.currentTarget.value)).toString();
                    if (newVal > this.searchState.after) {
                      this.invalidTimeWindow = false;
                      this.searchState.before = newVal;
                    } else {
                      this.invalidTimeWindow = true;
                    }
                  }}
                  defaultValue={dtBefore}
                  {...focusHandlers}
                />
              </FormGroup>
            </Layout>
          </Column>
          <Column cols={4}>
            <Layout position={Position.Relative} fullWidth={true} fullHeight={true}>
              <Layout padding={1} position={Position.Absolute} fullWidth={true} attachBottom={true}>
                <Button data-track-click="chatsearch-search" fullWidth={true} onClick={() => this.search()}>
                  Search
                </Button>
              </Layout>
            </Layout>
          </Column>
        </Grid>
      </StyledLayout>
    );
  }

  private enterHandler = (e: KeyboardEvent) => {
    // arrow function to preserve `this` as a reference to this component and not the event
    if (this.inputActive && e.code === "Enter") {
      this.search();
    }
  };

  private search() {
    this.props.chatStore!.update(this.searchState);
    const params = QueryString.stringify(this.searchState);
    this.props.history!.push({
      pathname: `/chatlogs/?${params}`
    });
  }
}

export const ChatSearch = withRouter(ChatSearchComponent);
