import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";
import { MockedProvider } from "react-apollo/test-utils";

import { DELETE_CLIPS, DeleteClipsButton, PublicProps, TestSelectors } from "aegis/features/delete-clips-button";
import { ConfirmationModalConfirmSelector } from "aegis/functionality/components/confirmation-modal/component";
import { mockErrorResponse, mockSuccessResponse } from "aegis/tests/apollo";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { waitSomeMs } from "aegis/tests/utils/wait-some-ms";
import { Button, CoreText } from "twitch-core-ui";

Enzyme.configure({ adapter: new Adapter() });

const mockDeleteClipSuccess1 = mockSuccessResponse(
  DELETE_CLIPS,
  { slugs: { slugs: ["success_slug1"] } },
  {
    data: {
      deleteClips: {
        success: [true],
        __typename: "ClipReport"
      }
    }
  }
);

const mockDeleteClipSuccess2 = mockSuccessResponse(
  DELETE_CLIPS,
  { slugs: { slugs: ["success_slug1", "success_slug2"] } },
  {
    data: {
      deleteClips: {
        success: [true, false],
        __typename: "ClipReport"
      }
    }
  }
);

const mockDeleteClipError = mockErrorResponse(DELETE_CLIPS, { slugs: { slugs: ["success_slug1", "error_slug"] } });

const mocks = [mockDeleteClipSuccess1, mockDeleteClipSuccess2, mockDeleteClipError];

function defaultPropsGenerator(overrides?: Partial<PublicProps>): PublicProps {
  return {
    slugs: [],
    onError: jest.fn(),
    onCompleted: jest.fn(),
    ...overrides
  };
}

function setupMount(overrides?: Partial<PublicProps>) {
  const props = defaultPropsGenerator();
  return Enzyme.mount(wrapWithGQLProvider(<DeleteClipsButton {...props} {...overrides} />, mocks).wrappedComponent);
}

describe("DeleteClipsButton", () => {
  const modalRoot = document.createElement("div");
  modalRoot.setAttribute("id", "portal-modal-root");
  const body = document.querySelector("body");
  if (body) {
    body.appendChild(modalRoot);
  }

  const globalDiv = document.createElement("div");
  document.body.appendChild(globalDiv);

  it("renders children", () => {
    const HELLO_WORLD = "HELLO WORLD";
    const wrapper = Enzyme.mount(
      <MockedProvider>
        <DeleteClipsButton slugs={[]}>
          <CoreText>{HELLO_WORLD}</CoreText>
        </DeleteClipsButton>
      </MockedProvider>
    );

    expect(wrapper.find(CoreText)).toHaveLength(1);
    expect(wrapper.text()).toContain(HELLO_WORLD);
  });

  it("disables button when loading gql mutation query", () => {
    const wrapper = setupMount();
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(false);
    wrapper
      .find({ "data-test-selector": TestSelectors.Button })
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(true);
  });

  it("disables button if given disabled prop", () => {
    const wrapper = setupMount({ disabled: true });
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(true);
  });

  it("calls onError prop if a query failed", async () => {
    const onError = jest.fn();
    const onCompleted = jest.fn();
    const wrapper = setupMount({
      slugs: ["success_slug1", "error_slug"],
      onError,
      onCompleted
    });

    wrapper
      .find({ "data-test-selector": TestSelectors.Button })
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");
    await waitSomeMs();
    wrapper.update();

    expect(onError).toHaveBeenCalledWith(Error("Network error: MockError: Unable to resolve GQL response"));
    expect(onCompleted).not.toHaveBeenCalled();
  });

  it("calls onCompleted prop if a query succeeded", async () => {
    const onError = jest.fn();
    const onCompleted = jest.fn();
    const wrapper = setupMount({
      slugs: ["success_slug1", "success_slug2"],
      onError,
      onCompleted
    });

    wrapper
      .find({ "data-test-selector": TestSelectors.Button })
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");
    await waitSomeMs();
    wrapper.update();

    expect(onCompleted).toHaveBeenCalledWith(mockDeleteClipSuccess2.result!.data);
    expect(onError).not.toHaveBeenCalled();
  });
});
