import * as React from "react";

import gql from "graphql-tag";
import { Mutation } from "react-apollo";

import { WithConfirmationModal } from "aegis/functionality/components/confirmation-modal";
import { Button, ButtonType } from "twitch-core-ui";

export interface PublicProps {
  slugs: string[];
  onCompleted?: (results: DeleteClipsResponse) => void;
  onError?: (error: Error) => void;
  disabled?: boolean;
  type?: ButtonType;
  children?: React.ReactNode;
}

export type DeleteClipsResponse = {
  deleteClips: {
    success: boolean[];
  };
};

export type InputProps = {
  slugs: {
    slugs: string[];
  };
};

export const DELETE_CLIPS = gql`
  mutation deleteClips($slugs: DeleteClipsInput!) {
    deleteClips(input: $slugs) {
      success
    }
  }
`;

export enum TestSelectors {
  Button = "delete-clips-button--button"
}

export class DeleteClipsMutation extends Mutation<DeleteClipsResponse, InputProps> {}

export class DeleteClipsButton extends React.Component<PublicProps> {
  public render() {
    const { onError, onCompleted, disabled, children, type, slugs } = this.props;
    const plural = slugs.length !== 1;
    const title = plural ? "Delete Clips" : "Delete Clip";
    const body = plural ? "Are you sure you want to delete these clips?" : "Are you sure you want to delete this clip?";
    return (
      <DeleteClipsMutation mutation={DELETE_CLIPS} onError={onError} onCompleted={onCompleted}>
        {(deleteClipsQuery, { loading }) => {
          return (
            <WithConfirmationModal
              title={title}
              body={body}
              confirmLabelOverride="Delete"
              onConfirm={() => {
                deleteClipsQuery({ variables: { slugs: { slugs } } });
              }}
            >
              {confirm => {
                return (
                  <Button
                    data-track-click="delete-clips-button"
                    onClick={confirm}
                    disabled={disabled !== undefined ? disabled : loading}
                    type={type}
                    children={children}
                    data-test-selector={TestSelectors.Button}
                  />
                );
              }}
            </WithConfirmationModal>
          );
        }}
      </DeleteClipsMutation>
    );
  }
}
