import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";
import { MockedProvider } from "react-apollo/test-utils";

import { ConfirmationModalConfirmSelector } from "aegis/functionality/components/confirmation-modal/component";
import { mockErrorResponse, mockSuccessResponse } from "aegis/tests/apollo";
import { wrapWithGQLProvider } from "aegis/tests/graphql";
import { waitSomeMs } from "aegis/tests/utils/wait-some-ms";
import { Button, CoreText } from "twitch-core-ui";
import { DELETE_VOD_COMMENT, DeleteVodCommentButton, PublicProps, TestSelectors } from "./component";

Enzyme.configure({ adapter: new Adapter() });

const mockDeleteCommentSuccess1 = mockSuccessResponse(
  DELETE_VOD_COMMENT,
  { id: "success" },
  {
    data: {
      deleteVodComment: {
        success: true,
        __typename: "DeleteVodCommentPayload"
      }
    }
  }
);

const mockDeleteCommentError = mockErrorResponse(DELETE_VOD_COMMENT, { id: "error" });

const mocks = [mockDeleteCommentSuccess1, mockDeleteCommentError];

function defaultPropsGenerator(overrides?: Partial<PublicProps>): PublicProps {
  return {
    id: "",
    onError: jest.fn(),
    onCompleted: jest.fn(),
    ...overrides
  };
}

function setupMount(overrides?: Partial<PublicProps>) {
  const props = defaultPropsGenerator();
  return Enzyme.mount(
    wrapWithGQLProvider(<DeleteVodCommentButton {...props} {...overrides} />, mocks).wrappedComponent
  );
}

describe("DeleteVodCommentButton", () => {
  const modalRoot = document.createElement("div");
  modalRoot.setAttribute("id", "portal-modal-root");
  const body = document.querySelector("body");
  if (body) {
    body.appendChild(modalRoot);
  }

  const globalDiv = document.createElement("div");
  document.body.appendChild(globalDiv);

  it("renders children", () => {
    const HELLO_WORLD = "HELLO WORLD";
    const wrapper = Enzyme.mount(
      <MockedProvider>
        <DeleteVodCommentButton id={""}>
          <CoreText>{HELLO_WORLD}</CoreText>
        </DeleteVodCommentButton>
      </MockedProvider>
    );

    expect(wrapper.find(CoreText)).toHaveLength(1);
    expect(wrapper.text()).toContain(HELLO_WORLD);
  });

  it("disables button when loading gql mutation query", () => {
    const wrapper = setupMount();
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(false);
    wrapper
      .find({ "data-test-selector": TestSelectors.Button })
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(true);
  });

  it("disables button if given disabled prop", () => {
    const wrapper = setupMount({ disabled: true });
    expect(
      wrapper
        .find({ "data-test-selector": TestSelectors.Button })
        .find(Button)
        .prop("disabled")
    ).toBe(true);
  });

  it("calls onError prop if a query failed", async () => {
    const onError = jest.fn();
    const onCompleted = jest.fn();
    const wrapper = setupMount({
      id: "error",
      onError,
      onCompleted
    });

    wrapper
      .find({ "data-test-selector": TestSelectors.Button })
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");
    await waitSomeMs();
    wrapper.update();

    expect(onError).toHaveBeenCalledWith(Error("Network error: MockError: Unable to resolve GQL response"));
    expect(onCompleted).not.toHaveBeenCalled();
  });

  it("calls onCompleted prop if a query succeeded", async () => {
    const onError = jest.fn();
    const onCompleted = jest.fn();
    const wrapper = setupMount({
      id: "success",
      onError,
      onCompleted
    });

    wrapper
      .find({ "data-test-selector": TestSelectors.Button })
      .find(Button)
      .simulate("click");
    wrapper
      .find({ "data-test-selector": ConfirmationModalConfirmSelector })
      .find(Button)
      .simulate("click");
    await waitSomeMs();
    wrapper.update();

    expect(onCompleted).toHaveBeenCalledWith(mockDeleteCommentSuccess1.result!.data);
    expect(onError).not.toHaveBeenCalled();
  });
});
