import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { Button, Radio, Select, TextArea } from "twitch-core-ui";
import { EmoteActionOption } from "../../models";
import { EmoticonApprovalForm, Props, State } from "./component";

Enzyme.configure({ adapter: new Adapter() });

const mockEmoteId = "522778";

const setupShallow = (overrides?: Partial<Props>): Enzyme.ShallowWrapper<Props, State> => {
  return Enzyme.shallow(
    <EmoticonApprovalForm
      defaultAction={EmoteActionOption.APPROVE}
      emoteId={mockEmoteId}
      onEmoticonAction={jest.fn()}
      onSubmit={jest.fn()}
      {...overrides}
    />
  );
};

const setupMount = (overrides?: Partial<Props>) => {
  return Enzyme.mount(
    <EmoticonApprovalForm
      defaultAction={EmoteActionOption.APPROVE}
      emoteId={mockEmoteId}
      onEmoticonAction={jest.fn()}
      onSubmit={jest.fn()}
      {...overrides}
    />
  );
};

describe("Emote Approval Form", () => {
  it("renders properly", () => {
    const wrapper = setupShallow();
    expect(wrapper).toMatchSnapshot();
  });

  it("renders the approve defer and decline radio buttons", () => {
    const wrapper = setupShallow();
    expect(wrapper.find(Radio)).toHaveLength(3);
    expect(
      wrapper
        .find(Radio)
        .first()
        .props().label
    ).toEqual("Approve");
    expect(
      wrapper
        .find(Radio)
        .at(1)
        .props().label
    ).toEqual("Decline");
    expect(
      wrapper
        .find(Radio)
        .at(2)
        .props().label
    ).toEqual("Defer");
  });

  it("renders the reject reason select", () => {
    const wrapper = setupShallow();
    expect(wrapper.find(Select)).toHaveLength(1);
  });

  it("renders the reject reason textarea", () => {
    const wrapper = setupShallow();
    expect(wrapper.find(TextArea)).toHaveLength(1);
  });

  it("renders the submit button", () => {
    const wrapper = setupShallow();
    expect(wrapper.find(Button)).toHaveLength(1);
  });

  it("calls the onEmoticonAction callback when clicking the submit button", () => {
    const wrapper = setupMount();
    wrapper.find(Button).simulate("click");
    expect(wrapper.props().onEmoticonAction).toHaveBeenCalledWith(mockEmoteId, EmoteActionOption.APPROVE, "", "");
  });

  it("calls the onEmoticonAction callback when changing the reject reason text", () => {
    const rejectReason = "Reject reason here";
    const rejectMessage = "more details about the reject";

    const wrapper = setupMount();

    wrapper.setState({
      action: EmoteActionOption.DECLINE,
      declineReason: rejectReason,
      declineMessage: rejectMessage
    });

    wrapper.find(TextArea).simulate("change", { currentTarget: { value: rejectReason } });

    expect(wrapper.props().onEmoticonAction).toHaveBeenLastCalledWith(
      mockEmoteId,
      EmoteActionOption.DECLINE,
      rejectReason,
      rejectMessage
    );
  });

  it("displays the error state when declined and there is no reason given", () => {
    const wrapper = setupShallow();
    wrapper.setState({
      action: EmoteActionOption.DECLINE
    });
    expect(wrapper.find(Select).props().error).toEqual(true);
  });

  it("does not display the error state when declined and there is a reason given", () => {
    const wrapper = setupShallow();
    wrapper.find(TextArea).simulate("change", { currentTarget: { value: "Reject reason here" } });
    wrapper.setState({
      action: EmoteActionOption.DECLINE
    });
    expect(wrapper.find(Select).props().error).toEqual(false);
  });
});
