import * as React from "react";

import { DeclineReason, declineReasons } from "aegis/features/emotes/decline-reasons";
import { EmoteActionOption } from "aegis/features/emotes/models";
import { CheckBoxFixer } from "aegis/functionality/components/checkbox-fixer";
import { Button, Layout, Radio, Select, TextArea } from "twitch-core-ui";

export interface Props {
  emoteId: string;
  defaultAction: EmoteActionOption;
  onEmoticonAction: (emoteId: string, action: EmoteActionOption, declineReason: string, declineMessage: string) => void;
  onSubmit: (emoteId: string) => void;
}

export interface State {
  declineReason: string;
  declineMessage: string;
  action: EmoteActionOption;
}

export class EmoticonApprovalForm extends React.Component<Props, State> {
  private form: HTMLFormElement;

  public state: State = {
    declineReason: "",
    declineMessage: "",
    action: this.props.defaultAction
  };

  public render() {
    const options: JSX.Element[] = [];
    options.push(
      <option key="default" value="default">
        Pick a Decline Reason
      </option>
    );
    declineReasons.forEach((value, key) => {
      options.push(
        <option key={key} value={key}>
          {value.title}
        </option>
      );
    });
    const declineSelect = (
      <Select error={this.isDeclineMessageRequired} onChange={this.onReasonSelect}>
        {options}
      </Select>
    );
    return (
      <Layout margin={{ x: 1 }}>
        <Layout>
          <CheckBoxFixer>
            <Radio
              onChange={() => this.onApprovalChange(EmoteActionOption.APPROVE)}
              checked={this.state.action === EmoteActionOption.APPROVE}
              name={`approve-decline-${this.props.emoteId}`}
              label="Approve"
            />
          </CheckBoxFixer>
          <CheckBoxFixer>
            <Radio
              onChange={() => this.onApprovalChange(EmoteActionOption.DECLINE)}
              checked={this.state.action === EmoteActionOption.DECLINE}
              name={`approve-decline-${this.props.emoteId}`}
              label="Decline"
            />
          </CheckBoxFixer>
          <Layout padding={{ y: 0.5 }} />
          <CheckBoxFixer>
            <Radio
              onChange={() => this.onApprovalChange(EmoteActionOption.DEFER)}
              checked={this.state.action === EmoteActionOption.DEFER}
              name={`approve-decline-${this.props.emoteId}`}
              label="Defer"
            />
          </CheckBoxFixer>
        </Layout>
        <form ref={this.getFormRef}>
          <Layout margin={{ y: 1 }}>{declineSelect}</Layout>
          <Layout margin={{ bottom: 1 }}>
            <TextArea
              error={this.isDeclineMessageRequired}
              onChange={this.onReasonChange}
              rows={5}
              placeholder={"Message in e-mail to the broadcaster"}
              value={this.state.declineMessage}
            />
          </Layout>
        </form>
        <Button
          data-track-click="emote-approval-form-submit"
          disabled={this.isDeclineMessageRequired}
          onClick={this.onSubmit}
        >
          Submit
        </Button>
      </Layout>
    );
  }

  private getFormRef = (instance: HTMLFormElement) => {
    this.form = instance;
  };

  private get isDeclineMessageRequired() {
    return this.state.action === EmoteActionOption.DECLINE && this.state.declineMessage.length < 1;
  }

  private onReasonSelect = (e: React.FormEvent<HTMLSelectElement>) => {
    const declineReason = declineReasons.get(e.currentTarget.value as DeclineReason);
    const message = declineReason ? declineReason.message : "";

    this.setState((prevState: State) => {
      let action = prevState.action;
      if (declineReason && prevState.declineReason === "") {
        action = EmoteActionOption.DECLINE;
      }
      return {
        declineMessage: message,
        declineReason: declineReason ? declineReason.title : "",
        action: action
      };
    }, this.updateEmoteStatus);
  };

  private onReasonChange = (e: React.FormEvent<HTMLTextAreaElement>) => {
    this.setState(
      {
        declineMessage: e.currentTarget.value
      },
      this.updateEmoteStatus
    );
  };

  private onApprovalChange = (action: EmoteActionOption) => {
    this.setState(
      (prevState: State) => {
        const declineMessage = action === EmoteActionOption.DECLINE ? prevState.declineMessage : "";
        return {
          action: action,
          declineMessage
        };
      },
      () => {
        this.form.reset();
        this.updateEmoteStatus();
      }
    );
  };

  private updateEmoteStatus = () => {
    const declineMessage = this.state.action === EmoteActionOption.DECLINE ? this.state.declineMessage : "";
    const declineReason = this.state.action === EmoteActionOption.DECLINE ? this.state.declineReason : "";
    this.props.onEmoticonAction(this.props.emoteId, this.state.action, declineReason, declineMessage);
  };

  private onSubmit = () => {
    this.updateEmoteStatus();
    this.props.onSubmit(this.props.emoteId);
  };
}
