import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { DeferredEmoteStore } from "aegis/stores";
import { CoreText, TableCell, TableHeading } from "twitch-core-ui";
import { mockEmoticonInfo } from "../../mocks";
import { EmoteActionOption, EmoticonStatus } from "../../models";
import { EmoticonApprovalForm } from "../emoticon-approval-form";
import { EmoticonRow, EmoticonRowProps, EmoticonsTableComponent, Props, TestSelectors } from "./component";

Enzyme.configure({ adapter: new Adapter() });

function defaultPropsGenerator(overrides?: Partial<Props>): Props {
  return {
    deferredEmoteStore: new DeferredEmoteStore(),
    pendingEmoticons: mockEmoticonInfo(),
    onEmoticonAction: jest.fn(),
    onSubmit: jest.fn(),
    updatePendingEmoticons: jest.fn(),
    ...overrides
  };
}

const setupShallow = (overrides?: Partial<Props>) => {
  const props = defaultPropsGenerator();
  return Enzyme.shallow(<EmoticonsTableComponent.wrappedComponent {...props} {...overrides} />);
};

describe("Emote approval table", () => {
  it("renders correctly the table headings", () => {
    const wrapper = setupShallow();
    expect(wrapper.find(TableHeading).length).toEqual(11);
  });

  it("renders the proper amount of rows", () => {
    const wrapper = setupShallow();
    expect(wrapper.find(EmoticonRow).length).toEqual(3);
  });

  [EmoteActionOption.APPROVE, EmoteActionOption.DEFER, EmoteActionOption.DECLINE].forEach(action => {
    it(`renders the proper cells with default ${action}`, () => {
      const emote = mockEmoticonInfo(1)[0];
      const props: EmoticonRowProps = {
        emote,
        defaultAction: action,
        onEmoticonAction: jest.fn(),
        onSubmit: jest.fn()
      };
      const wrapper = Enzyme.shallow(<EmoticonRow {...props} />);

      expect(wrapper.find(TableCell).length).toEqual(11);
      // emoteId
      expect(wrapper.find({ "data-test-selector": TestSelectors.IDColumn }).html()).toContain(emote.id);
      // ownerLogin
      expect(wrapper.find({ "data-test-selector": TestSelectors.OwnerColumn }).html()).toContain(emote.owner);
      // emote code
      expect(wrapper.find({ "data-test-selector": TestSelectors.CodeColumn }).html()).toContain(emote.code);
      // isPartner
      expect(
        wrapper
          .find({ "data-test-selector": TestSelectors.PartnerColumn })
          .find(CoreText)
          .html()
      ).toContain(true);
      // regex
      expect(wrapper.find({ "data-test-selector": TestSelectors.RegexColumn }).html()).toContain(emote.regex);
      // approval status
      expect(
        wrapper
          .find({ "data-test-selector": TestSelectors.StatusColumn })
          .find(CoreText)
          .html()
      ).toContain(EmoticonStatus.Pending);
      // img28
      expect(
        wrapper
          .find({ "data-test-selector": TestSelectors.Img28Column })
          .find({ src: `https://static-cdn.jtvnw.net/emoticons/v1/${emote.id}/1.0` }).length
      ).toEqual(1);
      // img56
      expect(
        wrapper
          .find({ "data-test-selector": TestSelectors.Img56Column })
          .find({ src: `https://static-cdn.jtvnw.net/emoticons/v1/${emote.id}/2.0` }).length
      ).toEqual(1);
      // img112
      expect(
        wrapper
          .find({ "data-test-selector": TestSelectors.Img112Column })
          .find({ src: `https://static-cdn.jtvnw.net/emoticons/v1/${emote.id}/3.0` }).length
      ).toEqual(1);
      // created on
      expect(
        wrapper
          .find({ "data-test-selector": TestSelectors.DateColumn })
          .find(CoreText)
          .html()
      ).toContain(emote.createdOn.toString());
      // form
      expect(
        wrapper.find({ "data-test-selector": TestSelectors.FormColumn }).find(EmoticonApprovalForm).length
      ).toEqual(1);
    });
  });
});
