import * as React from "react";

import { inject } from "mobx-react";

import { EmoticonApprovalForm } from "aegis/features/emotes/components/emoticon-approval-form";
import { EmoteActionOption, EmoticonInfo } from "aegis/features/emotes/models";
import { DeferredEmoteStoreProps } from "aegis/stores";

import {
  CoreText,
  Display,
  InjectLayout,
  Layout,
  SVG,
  SVGAsset,
  SVGType,
  Table,
  TableBody,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
  TextAlign,
  WordBreak
} from "twitch-core-ui";
import "./styles.scss";

export enum TestSelectors {
  ErrorMessage = "error-message",
  IDColumn = "id-column",
  OwnerColumn = "owner-column",
  CodeColumn = "code-column",
  PartnerColumn = "partner-column",
  RegexColumn = "regex-column",
  StatusColumn = "status-column",
  Img28Column = "img28-column",
  Img56Column = "img56-column",
  Img112Column = "img112-column",
  DateColumn = "date-column",
  FormColumn = "form-column"
}

export interface PublicProps {
  onEmoticonAction: (emoteId: string, action: EmoteActionOption, rejectReason: string, rejectMessage: string) => void;
  onSubmit: (emoteId: string) => void;
  updatePendingEmoticons: (pending: EmoticonInfo[]) => void;
  pendingEmoticons: EmoticonInfo[];
}

export type Props = PublicProps & DeferredEmoteStoreProps;

@inject("deferredEmoteStore")
export class EmoticonsTableComponent extends React.Component<Props> {
  // Declaring type for the wrapped component of the inject for tests to use
  public static wrappedComponent: React.ComponentClass<PublicProps>;

  public render() {
    const { deferredEmoteStore, pendingEmoticons, onEmoticonAction, onSubmit } = this.props;

    const rows: JSX.Element[] = pendingEmoticons.map(emote => (
      <EmoticonRow
        key={`emoticon-row-${emote.id}`}
        emote={emote}
        defaultAction={deferredEmoteStore!.isDeferred(emote.id) ? EmoteActionOption.DEFER : EmoteActionOption.APPROVE}
        onEmoticonAction={onEmoticonAction}
        onSubmit={onSubmit}
      />
    ));

    return (
      <Table>
        <TableHeader>
          <TableHeading label="ID" />
          <TableHeading label="Owner" />
          <TableHeading label="Emote Code" />
          <TableHeading label="Partner" />
          <TableHeading label="Regex" />
          <TableHeading label="Status" />
          <TableHeading label="28x28" />
          <TableHeading label="56x56" />
          <TableHeading label="112x112" />
          <TableHeading label="Created On" />
          <TableHeading label="" />
        </TableHeader>
        <TableBody>{rows}</TableBody>
      </Table>
    );
  }
}

export const EmoticonsTable: React.ComponentClass<PublicProps> = EmoticonsTableComponent;

export interface EmoticonRowProps {
  emote: EmoticonInfo;
  defaultAction: EmoteActionOption;
  onEmoticonAction: (emoteId: string, action: EmoteActionOption, rejectReason: string, rejectMessage: string) => void;
  onSubmit: (emoteId: string) => void;
}

export const EmoticonRow: React.SFC<EmoticonRowProps> = (props: EmoticonRowProps) => {
  return (
    <TableRow key={`pending-emote-${props.emote.id}`}>
      <TableCell data-test-selector={TestSelectors.IDColumn}>{props.emote.id}</TableCell>
      <TableCell data-test-selector={TestSelectors.OwnerColumn}>{props.emote.owner}</TableCell>
      <TableCell data-test-selector={TestSelectors.CodeColumn}>{props.emote.code}</TableCell>
      <TableCell data-test-selector={TestSelectors.PartnerColumn}>
        <Layout display={Display.Flex}>
          {props.emote.isPartner && <SVG type={SVGType.Brand} asset={SVGAsset.Verified} />}
          <InjectLayout padding={{ left: 0.5 }}>
            <CoreText>{props.emote.isPartner ? "true" : "false"}</CoreText>
          </InjectLayout>
        </Layout>
      </TableCell>
      <TableCell data-test-selector={TestSelectors.RegexColumn}>{props.emote.regex}</TableCell>
      <TableCell data-test-selector={TestSelectors.StatusColumn}>
        <Layout textAlign={TextAlign.Center} className="emoticon-approval__status-pending">
          <CoreText>{props.emote.status}</CoreText>
        </Layout>
      </TableCell>
      <TableCell data-test-selector={TestSelectors.Img28Column}>
        <EmoticonImage src={props.emote.img28Src} />
      </TableCell>
      <TableCell data-test-selector={TestSelectors.Img56Column}>
        <EmoticonImage src={props.emote.img56Src} />
      </TableCell>
      <TableCell data-test-selector={TestSelectors.Img112Column}>
        <EmoticonImage src={props.emote.img112Src} />
      </TableCell>
      <TableCell data-test-selector={TestSelectors.DateColumn}>
        <CoreText className="emoticon-approval__created-date" wordBreak={WordBreak.BreakWord}>
          {props.emote.createdOn.toString()}
        </CoreText>
      </TableCell>
      <TableCell data-test-selector={TestSelectors.FormColumn}>
        <EmoticonApprovalForm
          defaultAction={props.defaultAction}
          emoteId={props.emote.id}
          onEmoticonAction={props.onEmoticonAction}
          onSubmit={props.onSubmit}
        />
      </TableCell>
    </TableRow>
  );
};

const EmoticonImage: React.SFC<{ src: string }> = ({ src }) => {
  return <img src={src} className="emoticon-approval__image" />;
};
