import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { EmoticonsTable } from "aegis/features/emotes/components/emoticons-table";
import { mockEmoticonInfo } from "aegis/features/emotes/mocks";
import { EmoteActionOption, EmoticonAction, EmoticonApprovalError } from "aegis/features/emotes/models";
import { DeferredEmoteStore } from "aegis/stores";
import { Button } from "twitch-core-ui";
import { PendingEmoticonSubmitComponent, Props, TestSelectors } from "./component";

Enzyme.configure({ adapter: new Adapter() });

function defaultPropsGenerator(overrides?: Partial<Props>): Props {
  return {
    pendingEmoticons: mockEmoticonInfo(),
    deferredEmoteStore: new DeferredEmoteStore(),
    onSubmitEmoticonActions: jest.fn(() => Promise.resolve(EmoticonApprovalError.None)),
    mutationLoading: false,
    ...overrides
  };
}

const setupShallow = (overrides?: Partial<Props>) => {
  const props = defaultPropsGenerator(overrides);
  return Enzyme.shallow(<PendingEmoticonSubmitComponent.wrappedComponent {...props} />);
};

describe("PendingEmoticonSubmitComponent", () => {
  it("renders properly", () => {
    const wrapper = setupShallow();
    expect(wrapper.find(EmoticonsTable).length).toBe(1);
    expect(wrapper.find(Button).length).toBe(2);
  });

  it("does not render the error message if there are only approved emotes", () => {
    const pendingEmoticons = mockEmoticonInfo(1);
    const wrapper = setupShallow({ pendingEmoticons });

    const pending = new Map<string, EmoticonAction>();
    pending.set(pendingEmoticons[0].id, {
      emoticonID: pendingEmoticons[0].id,
      action: EmoteActionOption.APPROVE,
      reason: "",
      detailedReason: ""
    });
    wrapper.setState({
      pendingEmoteActions: pending
    });
    wrapper
      .find(Button)
      .first()
      .simulate("click");
    expect(wrapper.find({ "data-test-selector": TestSelectors.ErrorMessage })).toHaveLength(0);
  });

  it("does not submit deferred buttons", () => {
    const onSubmitEmoticonActions = jest.fn(() => Promise.resolve(EmoticonApprovalError.None));
    const pendingEmoticons = mockEmoticonInfo(1);
    const deferredEmoteStore = new DeferredEmoteStore();
    const wrapper = setupShallow({ pendingEmoticons, deferredEmoteStore, onSubmitEmoticonActions });

    expect(wrapper.find({ "data-test-selector": TestSelectors.ErrorMessage })).toHaveLength(0);

    const pending = new Map<string, EmoticonAction>();
    pending.set(pendingEmoticons[0].id, {
      emoticonID: pendingEmoticons[0].id,
      action: EmoteActionOption.DEFER,
      reason: "",
      detailedReason: ""
    });
    wrapper.setState({
      pendingEmoteActions: pending
    });

    expect(deferredEmoteStore.isDeferred(pendingEmoticons[0].id)).toBeFalsy();

    wrapper
      .find(Button)
      .first()
      .simulate("click");

    expect(wrapper.find({ "data-test-selector": TestSelectors.ErrorMessage })).toHaveLength(0);
    expect(deferredEmoteStore.isDeferred(pendingEmoticons[0].id)).toBeTruthy();
    expect(onSubmitEmoticonActions).toHaveBeenCalledWith([]);
  });

  it("does not render the error messsage if there are unapproved emotes with reasons", () => {
    const pendingEmoticons = mockEmoticonInfo(1);
    const wrapper = setupShallow({ pendingEmoticons });
    expect(wrapper.find({ "data-test-selector": TestSelectors.ErrorMessage })).toHaveLength(0);

    const pending = new Map<string, EmoticonAction>();
    pending.set(pendingEmoticons[0].id, {
      emoticonID: pendingEmoticons[0].id,
      action: EmoteActionOption.DECLINE,
      reason: "some reason",
      detailedReason: "here is more info about the rejection"
    });
    wrapper.setState({
      pendingEmoteActions: pending
    });
    wrapper
      .find(Button)
      .first()
      .simulate("click");
    expect(wrapper.find({ "data-test-selector": TestSelectors.ErrorMessage })).toHaveLength(0);
  });

  it("renders the error message if there are unapproved emotes without reason", () => {
    const pendingEmoticons = mockEmoticonInfo(1);
    const wrapper = setupShallow({ pendingEmoticons });
    expect(wrapper.find({ "data-test-selector": TestSelectors.ErrorMessage })).toHaveLength(0);

    const pending = new Map<string, EmoticonAction>();
    pending.set(pendingEmoticons[0].id, {
      emoticonID: pendingEmoticons[0].id,
      action: EmoteActionOption.DECLINE,
      reason: "",
      detailedReason: ""
    });
    wrapper.setState({
      pendingEmoteActions: pending
    });
    wrapper
      .find(Button)
      .first()
      .simulate("click");
    expect(wrapper.find({ "data-test-selector": TestSelectors.ErrorMessage })).toHaveLength(1);
  });
});
