import * as React from "react";

import { inject } from "mobx-react";

import { EmoticonsTable } from "aegis/features/emotes/components/emoticons-table";
import { EmoteActionOption, EmoticonAction, EmoticonApprovalError, EmoticonInfo } from "aegis/features/emotes/models";
import { DeferredEmoteStoreProps } from "aegis/stores";

import { Button, ButtonState, Color, CoreText, Display, InjectLayout, JustifyContent, Layout } from "twitch-core-ui";

export enum TestSelectors {
  ErrorMessage = "error-message"
}

export interface PublicProps {
  pendingEmoticons: EmoticonInfo[];
  onSubmitEmoticonActions: (actions: EmoticonAction[]) => Promise<EmoticonApprovalError>;
  mutationLoading: boolean;
}

export interface State {
  pendingEmoteActions: Map<string, EmoticonAction>;
  error: EmoticonApprovalError;
}

export type Props = PublicProps & DeferredEmoteStoreProps;

@inject("deferredEmoteStore")
export class PendingEmoticonSubmitComponent extends React.Component<Props, State> {
  // Declaring type for the wrapped component of the inject for tests to use
  public static wrappedComponent: React.ComponentClass<Props, State>;

  public state: State = {
    pendingEmoteActions: new Map<string, EmoticonAction>(),
    error: EmoticonApprovalError.None
  };

  public componentWillReceiveProps(nextProps: PublicProps) {
    if (nextProps.pendingEmoticons !== this.props.pendingEmoticons) {
      this.updatePendingEmoticons(nextProps.pendingEmoticons);
    }
  }

  public render() {
    return (
      <Layout>
        <Layout padding={1} display={Display.Flex} fullWidth justifyContent={JustifyContent.End}>
          {this.state.error !== EmoticonApprovalError.None && (
            <InjectLayout margin={{ x: 1 }}>
              <CoreText data-test-selector={TestSelectors.ErrorMessage} color={Color.Error}>
                {this.errorMessage}
              </CoreText>
            </InjectLayout>
          )}
          <Button
            data-track-click="emote-approval-submit-all-page-begin"
            state={this.props.mutationLoading ? ButtonState.Loading : ButtonState.Default}
            onClick={this.onSubmitAll}
          >
            Submit All
          </Button>
        </Layout>
        <EmoticonsTable
          pendingEmoticons={this.props.pendingEmoticons}
          onEmoticonAction={this.onEmoticonAction}
          onSubmit={this.onSubmitSingle}
          updatePendingEmoticons={this.updatePendingEmoticons}
        />
        <Layout padding={1} display={Display.Flex} fullWidth justifyContent={JustifyContent.End}>
          <Button
            data-track-click="emote-approval-submit-all-page-end"
            state={this.props.mutationLoading ? ButtonState.Loading : ButtonState.Default}
            onClick={this.onSubmitAll}
          >
            Submit All
          </Button>
        </Layout>
      </Layout>
    );
  }

  private onEmoticonAction = (
    emoticonID: string,
    actionOption: EmoteActionOption,
    reason: string,
    detailedReason: string
  ) => {
    const action: EmoticonAction = {
      emoticonID,
      action: actionOption,
      reason,
      detailedReason
    };
    this.state.pendingEmoteActions.set(emoticonID, action);
  };

  private onSubmitSingle = (emoteId: string) => {
    this.submitEmoteActions(emoteId);
  };

  private updatePendingEmoticons = (pending: EmoticonInfo[]) => {
    const { deferredEmoteStore } = this.props;
    const pendingActions = new Map<string, EmoticonAction>();
    pending.forEach(pendingEmote => {
      pendingActions.set(pendingEmote.id, {
        emoticonID: pendingEmote.id,
        action: deferredEmoteStore!.isDeferred(pendingEmote.id) ? EmoteActionOption.DEFER : EmoteActionOption.APPROVE,
        reason: "",
        detailedReason: ""
      });
    });
    this.setState({
      pendingEmoteActions: pendingActions
    });
  };

  private onSubmitAll = () => {
    if (this.isRejectionReasonMissing()) {
      this.setState({
        error: EmoticonApprovalError.MissingRejectReason
      });
    } else {
      this.setState({
        error: EmoticonApprovalError.None
      });
      this.submitEmoteActions();
    }
    window.scrollTo({
      top: 0
    });
  };

  private isRejectionReasonMissing() {
    return Array.from(this.state.pendingEmoteActions.keys()).some(emoteId => {
      const action = this.state.pendingEmoteActions.get(emoteId);
      if (action && action.action === EmoteActionOption.DECLINE && (!action.reason || action.reason === "")) {
        return true;
      }
      return false;
    });
  }

  private submitEmoteActions(emoteId?: string) {
    let actions: EmoticonAction[] = [];
    if (emoteId) {
      actions = [this.state.pendingEmoteActions.get(emoteId)!];
    } else {
      actions = Array.from(this.state.pendingEmoteActions).map(pair => pair[1]);
    }

    this.props.deferredEmoteStore!.updateDeferred(
      actions.filter(a => a.action === EmoteActionOption.DEFER).map(a => a.emoticonID)
    );
    actions = actions.filter(a => a.action !== EmoteActionOption.DEFER);

    this.props.onSubmitEmoticonActions(actions).then(error => {
      if (error !== EmoticonApprovalError.None) {
        this.setState({
          error
        });
      }
    });
  }

  private get errorMessage(): string {
    let message = "";
    switch (this.state.error) {
      case EmoticonApprovalError.MissingRejectReason:
        message =
          "One or more of the declined emoticons is missing a reason. Please add a decline reason and try again.";
        break;
      case EmoticonApprovalError.RequestError:
        message = "There was an error making the emote approval request. Please try again later.";
        break;
      default:
        message = "There was an error submitting the emote approvals. Please try again later.";
    }
    return message;
  }
}
