import * as React from "react";

import gql from "graphql-tag";
import { compose, graphql, GraphqlQueryControls, withApollo } from "react-apollo";

import {
  EmoteActionOption,
  EmoticonAction,
  EmoticonApprovalError,
  EmoticonApprovalPartnerFilter,
  EmoticonInfo,
  EmoticonStatus,
  PendingEmoticonsResponse
} from "aegis/features/emotes/models";
import { UserRole } from "aegis/models";
import { ApolloClient } from "apollo-client";
import { Color, CoreText } from "twitch-core-ui";
import { PendingEmoticonSubmitComponent } from "./component";

const DEFAULT_EMOTES_PER_PAGE = 15;

export interface PublicProps {
  currentBucket: number;
  bucketCount: number;
  partnerFilter: EmoticonApprovalPartnerFilter;
}

export type Props = PublicProps & GraphQLProps & ApolloProps;

interface GraphQLProps {
  data: PendingEmoticonsResponse & GraphqlQueryControls;
}

interface ApolloProps {
  client?: ApolloClient<{}>;
}

interface State {
  mutationLoading: boolean;
}

interface ReviewAction {
  emoticonID: string;
  isApproved: boolean;
  reason: string;
  detailedReason: string;
}

export class PendingEmoticonSubmitContainer extends React.Component<Props, State> {
  public state: State = {
    mutationLoading: false
  };

  public render() {
    const { pendingEmoticons, error } = this.props.data;

    if (error) {
      console.error("An error occurred", error);
      return (
        <CoreText color={Color.Error}>
          The following error occurred: {error.message}. Please refresh the page. If this continues to happen, please
          page safety oncall.
        </CoreText>
      );
    }

    const pending: EmoticonInfo[] = pendingEmoticons
      ? pendingEmoticons.edges.map(edge => ({
          id: edge.node.id,
          owner: edge.node.owner.login,
          code: edge.node.code,
          isPartner: edge.node.owner.roles.some(role => role === UserRole.Partner),
          regex: `${edge.node.prefix}${edge.node.code}`,
          status: EmoticonStatus.Pending,
          img28Src: `https://static-cdn.jtvnw.net/emoticons/v1/${edge.node.id}/1.0`,
          img56Src: `https://static-cdn.jtvnw.net/emoticons/v1/${edge.node.id}/2.0`,
          img112Src: `https://static-cdn.jtvnw.net/emoticons/v1/${edge.node.id}/3.0`,
          createdOn: new Date(edge.node.uploadTime)
        }))
      : [];
    return (
      <div>
        <PendingEmoticonSubmitComponent
          pendingEmoticons={pending}
          onSubmitEmoticonActions={this.onSubmitEmoticonActions}
          mutationLoading={this.state.mutationLoading}
        />
      </div>
    );
  }

  private onSubmitEmoticonActions = async (actions: EmoticonAction[]): Promise<EmoticonApprovalError> => {
    const reviewActions: ReviewAction[] = actions.map(a => {
      return {
        emoticonID: a.emoticonID,
        isApproved: a.action === EmoteActionOption.APPROVE,
        reason: a.reason,
        detailedReason: a.detailedReason
      };
    });
    try {
      this.setState({
        mutationLoading: true
      });
      await this.props.client!.mutate({
        mutation: ReviewEmotesMutation,
        refetchQueries: ["pendingEmotes"],
        variables: {
          input: {
            reviews: reviewActions
          }
        }
      });
      return EmoticonApprovalError.None;
    } catch (error) {
      return EmoticonApprovalError.RequestError;
    } finally {
      this.setState({
        mutationLoading: false
      });
    }
  };
}

const ReviewEmotesMutation = gql`
  mutation reviewEmoticons($input: ReviewPendingEmoticonsInput!) {
    reviewPendingEmoticons(input: $input) {
      emoticonIDs
    }
  }
`;

const PendingEmotesQuery = gql`
  query pendingEmotes($first: Int, $after: Cursor, $emoteBucket: Int, $bucketCount: Int, $filterBy: EmoticonFilter) {
    pendingEmoticons(
      first: $first
      after: $after
      emoteBucket: $emoteBucket
      bucketCount: $bucketCount
      filterBy: $filterBy
    ) {
      edges {
        node {
          id
          prefix
          code
          uploadTime
          owner {
            id
            login
            roles
          }
        }
      }
    }
  }
`;

export const PendingEmoticonSubmit: React.ComponentClass<PublicProps> = compose(
  withApollo,
  graphql<PublicProps, PendingEmoticonsResponse, {}>(PendingEmotesQuery, {
    options: (props: PublicProps) => {
      return {
        fetchPolicy: "network-only",
        variables: {
          first: DEFAULT_EMOTES_PER_PAGE,
          after: null,
          emoteBucket: props.currentBucket,
          bucketCount: props.bucketCount,
          filterBy: props.partnerFilter
        }
      };
    }
  })
)(PendingEmoticonSubmitContainer);
