import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { PendingEmoticonSubmit } from "aegis/features/emotes/components/pending-emoticon-submit";
import { EmoticonApprovalPartnerFilter } from "aegis/features/emotes/models";
import { FormGroup } from "twitch-core-ui";
import {
  ALL_BUCKET,
  ALL_BUCKET_STATE,
  DEFAULT_BUCKET_COUNT,
  DEFAULT_BUCKET_STATE,
  getBucketState,
  PendingEmoticons,
  PublicProps
} from "./component";

Enzyme.configure({ adapter: new Adapter() });

const setupShallow = (): Enzyme.ShallowWrapper<PublicProps> => {
  return Enzyme.shallow(<PendingEmoticons />);
};

describe("PendingEmoticons", () => {
  it("renders correctly", () => {
    const wrapper = setupShallow();
    expect(wrapper.find(FormGroup).find({ label: "Bucket" }).length).toBe(1);
    expect(wrapper.find(FormGroup).find({ label: "Filter" }).length).toBe(1);
    expect(wrapper.find(PendingEmoticonSubmit).length).toBe(1);
  });

  it("updates state when the Bucket select is changed", () => {
    const wrapper = setupShallow();
    wrapper
      .find(FormGroup)
      .find({ label: "Bucket" })
      .childAt(0)
      .props()
      .onChange({ currentTarget: { value: "2" } });
    expect(wrapper.state("currentBucket")).toBe(2);
    expect(wrapper.state("bucketCount")).toBe(DEFAULT_BUCKET_COUNT);
  });

  it("updates state when the Partner Filter select is changed", () => {
    const wrapper = setupShallow();
    wrapper
      .find(FormGroup)
      .find({ label: "Filter" })
      .childAt(0)
      .props()
      .onChange({ currentTarget: { value: EmoticonApprovalPartnerFilter.Affiliate } });
    expect(wrapper.state("partnerFilter")).toBe(EmoticonApprovalPartnerFilter.Affiliate);
  });

  it("updates state correctly when the All bucket is selected", () => {
    const wrapper = setupShallow();
    wrapper
      .find(FormGroup)
      .find({ label: "Bucket" })
      .childAt(0)
      .props()
      .onChange({ currentTarget: { value: `${ALL_BUCKET}` } });
    expect(wrapper.state("currentBucket")).toBe(0);
    expect(wrapper.state("bucketCount")).toBe(1);
  });

  it("passes the partner filter as a string enum down to the submit component", () => {
    const wrapper = setupShallow();
    wrapper.setState({
      partnerFilter: EmoticonApprovalPartnerFilter.Affiliate
    });
    expect(wrapper.find(PendingEmoticonSubmit).props().partnerFilter).toEqual("affiliate");
  });

  describe("getBucketState function", () => {
    it("returns the all bucket state when 'ALL' is passed in", () => {
      const returnVal = getBucketState(ALL_BUCKET);
      expect(returnVal).toEqual(ALL_BUCKET_STATE);
    });

    it("returns state with the proper currentBucket and bucketCount values when a valid bucket number is passed in", () => {
      const returnVal = getBucketState("1");
      expect(returnVal.currentBucket).toEqual(1);
      expect(returnVal.bucketCount).toEqual(DEFAULT_BUCKET_COUNT);
    });

    it("returns the default bucket state if an invalid bucket number is passed in", () => {
      const returnVal = getBucketState("Kappa");
      expect(returnVal.currentBucket).toEqual(DEFAULT_BUCKET_STATE.currentBucket);
      expect(returnVal.bucketCount).toEqual(DEFAULT_BUCKET_STATE.bucketCount);
    });
  });
});
