import * as React from "react";

import * as QueryString from "query-string";
import { RouteComponentProps, withRouter } from "react-router";

import {
  AlignItems,
  Button,
  ButtonType,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  Input,
  InputType,
  JustifyContent,
  Layout
} from "twitch-core-ui";
import { EnforcementView } from "../enforcement/container";

export interface EnforcementSearchQueryProps {
  user?: string;
  admin?: string;
}
export interface State {
  user?: string;
  admin?: string;
  inputUser: string;
  inputAdmin: string;
}
type Props = RouteComponentProps<{}> & EnforcementSearchQueryProps;

export class EnforcementSearchComponent extends React.Component<Props, State> {
  constructor(props: Props) {
    super(props);

    this.state = {
      inputUser: "",
      inputAdmin: "",
      user: props.user,
      admin: props.admin
    };
  }

  public render() {
    return (
      <Layout
        display={Display.Flex}
        justifyContent={JustifyContent.Center}
        flexDirection={FlexDirection.Column}
        padding={{ top: 2 }}
      >
        <Layout margin={{ left: 2 }}>
          <CoreText fontSize={FontSize.Size2} bold>
            Enforcement Search
          </CoreText>
          <CoreText fontSize={FontSize.Size5}>
            Search by user or by admin to view all suspensions received/given, or both to view all suspensions given to
            a user by an admin.
          </CoreText>
        </Layout>
        <Layout display={Display.Flex}>
          <Layout display={Display.Flex} margin={{ top: 1, left: 2 }} alignItems={AlignItems.End}>
            <Layout margin={{ right: 0.5 }} display={Display.Flex} flexDirection={FlexDirection.Column}>
              <Layout margin={{ bottom: 0.5 }}>
                <CoreText bold>Target User Login or ID</CoreText>
              </Layout>
              <Input
                type={InputType.Text}
                placeholder={"User"}
                onChange={this.updateUser}
                onKeyDown={this.onKeyDown}
                defaultValue={this.props.user}
              />
            </Layout>
            <Layout margin={{ right: 0.5 }} display={Display.Flex} flexDirection={FlexDirection.Column}>
              <Layout margin={{ bottom: 0.5 }}>
                <CoreText bold>Admin LDAP ID</CoreText>
              </Layout>
              <Input
                type={InputType.Text}
                placeholder={"Admin"}
                defaultValue={this.props.admin}
                onChange={this.updateAdmin}
                onKeyDown={this.onKeyDown}
              />
            </Layout>
            <Layout flexGrow={0}>
              <Button type={ButtonType.Default} title="Search" onClick={this.search}>
                Search
              </Button>
            </Layout>
          </Layout>
        </Layout>
        <Layout margin={{ x: 1 }}>
          <EnforcementView adminLDAP={this.state.admin} user={this.state.user} />
        </Layout>
      </Layout>
    );
  }
  private updateUser: React.FormEventHandler<HTMLInputElement> = e => {
    this.setState({
      inputUser: e.currentTarget.value
    });
  };

  private updateAdmin: React.FormEventHandler<HTMLInputElement> = e => {
    this.setState({
      inputAdmin: e.currentTarget.value
    });
  };

  private search = () => {
    this.setState(state => ({
      admin: state.inputAdmin,
      user: state.inputUser
    }));
    const params = QueryString.stringify({ admin: this.state.inputAdmin, user: this.state.inputUser });
    this.props.history.push({ pathname: "/enforcementsearch", search: params });
  };

  private onKeyDown = (e: React.KeyboardEvent<HTMLInputElement>) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault();
      this.search();
    }
  };
}

export const EnforcementSearch: React.ComponentClass<EnforcementSearchQueryProps> = withRouter<Props>(
  EnforcementSearchComponent
);
