import * as React from "react";

import "aegis/functionality/date-extensions";
import { MockEnforcement, MockSuspension } from "aegis/models/mocks/enforcements";
import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";
import { EnforcementSuspensionBody, TestSelectors } from "./component";

Enzyme.configure({ adapter: new Adapter() });

describe("SuspensionBody", () => {
  it("Displays permanent for ip ban", () => {
    const enforcement = MockEnforcement({
      createdAt: "2018-05-11T18:29:49Z",
      updatedAt: "2018-05-11T18:29:49Z",
      suspension: MockSuspension({
        createdAt: "2018-05-11T18:29:49Z",
        updatedAt: "2018-05-11T18:29:49Z",
        ipBan: -1
      })
    });
    const wrapper = Enzyme.mount(<EnforcementSuspensionBody enforcement={enforcement} />);
    expect(wrapper).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.IPBan }).text()).toBe("Permanent");
  });

  it("Displays the length for temporary bans", () => {
    const enforcement = MockEnforcement({
      createdAt: "2018-05-11T18:29:49Z",
      updatedAt: "2018-05-11T18:29:49Z",
      suspension: MockSuspension({
        createdAt: "2018-05-11T18:29:49Z",
        updatedAt: "2018-05-11T18:29:49Z",
        ipBan: 86400
      })
    });
    const wrapper = Enzyme.mount(<EnforcementSuspensionBody enforcement={enforcement} />);
    expect(wrapper).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.IPBan }).text()).toBe("24h");
  });

  it("Displays false when there is no IPBan", () => {
    const enforcement = MockEnforcement({
      createdAt: "2018-05-11T18:29:49Z",
      updatedAt: "2018-05-11T18:29:49Z",
      suspension: MockSuspension({
        createdAt: "2018-05-11T18:29:49Z",
        updatedAt: "2018-05-11T18:29:49Z",
        ipBan: 0
      })
    });
    const wrapper = Enzyme.mount(<EnforcementSuspensionBody enforcement={enforcement} />);
    expect(wrapper).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.IPBan }).text()).toBe("False");
  });

  it("Displays not found when no ip ban is found", () => {
    const enforcement = MockEnforcement({
      createdAt: "2018-05-11T18:29:49Z",
      updatedAt: "2018-05-11T18:29:49Z",
      suspension: MockSuspension({
        createdAt: "2018-05-11T18:29:49Z",
        updatedAt: "2018-05-11T18:29:49Z",
        ipBan: undefined
      })
    });
    const wrapper = Enzyme.mount(<EnforcementSuspensionBody enforcement={enforcement} />);
    expect(wrapper).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.IPBan }).text()).toBe("Not Found");
  });

  it("Displays Permanent duration when permanent flag is set", () => {
    const enforcement = MockEnforcement({
      createdAt: "2018-05-11T18:29:49Z",
      updatedAt: "2018-05-11T18:29:49Z",
      suspension: MockSuspension({
        createdAt: "2018-05-11T18:29:49Z",
        updatedAt: "2018-05-11T18:29:49Z",
        duration: undefined,
        permanent: true
      })
    });
    const wrapper = Enzyme.mount(<EnforcementSuspensionBody enforcement={enforcement} />);
    expect(wrapper).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.Duration }).text()).toBe("Permanent");
  });

  it("Displays length when duration is a number", () => {
    const enforcement = MockEnforcement({
      createdAt: "2018-05-11T18:29:49Z",
      updatedAt: "2018-05-11T18:29:49Z",
      suspension: MockSuspension({
        createdAt: "2018-05-11T18:29:49Z",
        updatedAt: "2018-05-11T18:29:49Z",
        duration: 86400
      })
    });
    const wrapper = Enzyme.mount(<EnforcementSuspensionBody enforcement={enforcement} />);
    expect(wrapper).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.Duration }).text()).toBe("24h");
  });

  it("Displays Not Found when duration is missing", () => {
    const enforcement = MockEnforcement({
      createdAt: "2018-05-11T18:29:49Z",
      updatedAt: "2018-05-11T18:29:49Z",
      suspension: MockSuspension({
        createdAt: "2018-05-11T18:29:49Z",
        updatedAt: "2018-05-11T18:29:49Z",
        permanent: false,
        duration: undefined
      })
    });
    const wrapper = Enzyme.mount(<EnforcementSuspensionBody enforcement={enforcement} />);
    expect(wrapper).toBeTruthy();
    expect(wrapper.find({ "data-test-selector": TestSelectors.Duration }).text()).toBe("Not Found");
  });
});
