import * as React from "react";

import { CoreLink, CoreText, Display, FlexDirection, JustifyContent, Layout, StyledLayout } from "twitch-core-ui";

import { formatDurationTimestamp } from "aegis/features/report-view/components/report-body/stream-embed/component";
import { UserCardLink } from "aegis/features/user-card/user-link";
import { UserRoleTags } from "aegis/features/user-role-tags";
import { Enforcement, UserRole } from "aegis/models";
import { EnforcementColumn, EnforcementRow } from "../component";
import { Description } from "../description";

export interface SuspensionProps {
  enforcement: Enforcement;
  roles?: UserRole[];
}

export enum TestSelectors {
  IPBan = "suspension-body-IP-ban",
  Duration = "suspension-body-duration"
}

export const EnforcementSuspensionBody: React.SFC<SuspensionProps> = ({ enforcement, roles }) => {
  let IPBan;
  // Non strict check on null to check both undefined and null
  if (enforcement.suspension && enforcement.suspension.ipBan != null) {
    if (enforcement.suspension.ipBan === -1 || enforcement.suspension.ipBan === true) {
      IPBan = "Permanent";
      // Falsy check here checks if ipBan is either 0 or false
    } else if (!enforcement.suspension.ipBan) {
      IPBan = "False";
    } else {
      IPBan = formatDurationTimestamp(enforcement.suspension.ipBan);
    }
  } else {
    IPBan = "Not Found";
  }

  let duration = "Not Found";
  if (enforcement.suspension) {
    if (enforcement.suspension.duration) {
      duration = formatDurationTimestamp(enforcement.suspension.duration);
    } else if (enforcement.suspension.permanent === true) {
      duration = "Permanent";
    }
  }
  return (
    <StyledLayout>
      <Layout display={Display.Flex} justifyContent={JustifyContent.Between}>
        <EnforcementColumn header="Enforcement Metadata">
          <StyledLayout borderBottom flexGrow={1} fullWidth />
          <Layout display={Display.Flex}>
            <Layout display={Display.Flex} flexDirection={FlexDirection.Column} margin={{ right: 2 }}>
              <EnforcementRow label>Suspension ID</EnforcementRow>
              <EnforcementRow label>User</EnforcementRow>
              <EnforcementRow label>From</EnforcementRow>
              <EnforcementRow label>Created at</EnforcementRow>
              <EnforcementRow label>Updated at</EnforcementRow>
            </Layout>
            <Layout display={Display.Flex} flexDirection={FlexDirection.Column} margin={{ right: 2 }}>
              <EnforcementRow>{enforcement.id}</EnforcementRow>
              <Layout margin={{ y: 0.5 }} display={Display.Flex}>
                <UserCardLink userID={enforcement.targetUserID}>
                  <CoreLink>
                    <CoreText>
                      {!!enforcement.suspension ? enforcement.suspension.targetUserUsername : ""} [{
                        enforcement.targetUserID
                      }]
                    </CoreText>
                  </CoreLink>
                </UserCardLink>
                <Layout margin={{ left: 0.5 }}>
                  <UserRoleTags roles={roles} />
                </Layout>
              </Layout>
              <EnforcementRow>{enforcement.fromUserID}</EnforcementRow>
              <EnforcementRow>{new Date(enforcement.createdAt).toDisplayFormat()}</EnforcementRow>
              <EnforcementRow>{new Date(enforcement.updatedAt).toDisplayFormat()}</EnforcementRow>
            </Layout>
          </Layout>
        </EnforcementColumn>
        <EnforcementColumn header="Violation Details">
          <StyledLayout borderBottom flexGrow={1} />
          <Layout display={Display.Flex}>
            <Layout display={Display.Flex} flexDirection={FlexDirection.Column} margin={{ right: 2 }}>
              <EnforcementRow label>Content</EnforcementRow>
              <EnforcementRow label>Category</EnforcementRow>
              <EnforcementRow label>Reason</EnforcementRow>
              <EnforcementRow label>Detailed Reason</EnforcementRow>
            </Layout>

            <Layout display={Display.Flex} flexDirection={FlexDirection.Column} margin={{ right: 2 }}>
              <EnforcementRow>{enforcement.content}</EnforcementRow>
              <EnforcementRow>
                {!!enforcement.suspension && !!enforcement.suspension.targetUserGame
                  ? enforcement.suspension.targetUserGame
                  : "No Category found"}
              </EnforcementRow>
              <EnforcementRow>{enforcement.reason}</EnforcementRow>
              <EnforcementRow>{enforcement.detailedReason}</EnforcementRow>
            </Layout>
          </Layout>
        </EnforcementColumn>
        <EnforcementColumn header="Punishment Details">
          <StyledLayout borderBottom flexGrow={1} fullWidth />
          <Layout display={Display.Flex}>
            <Layout display={Display.Flex} flexDirection={FlexDirection.Column} margin={{ right: 2 }}>
              <EnforcementRow label>Duration</EnforcementRow>
              <EnforcementRow label>IP Ban</EnforcementRow>
              <EnforcementRow label>Cleared channel images</EnforcementRow>
            </Layout>

            <Layout display={Display.Flex} flexDirection={FlexDirection.Column} margin={{ right: 2 }}>
              <EnforcementRow data-test-selector={TestSelectors.Duration}>{duration}</EnforcementRow>
              <EnforcementRow data-test-selector={TestSelectors.IPBan}>{IPBan}</EnforcementRow>
              <EnforcementRow>
                {!!enforcement.suspension
                  ? enforcement.suspension.clearedChannelImages.valueOf().toString()
                  : "Cleared channel images not found"}
              </EnforcementRow>
            </Layout>
          </Layout>
        </EnforcementColumn>
      </Layout>
      <Layout margin={{ y: 2, x: 2 }}>
        <Description
          id={enforcement.id}
          description={
            !!enforcement.suspension && !!enforcement.suspension.description
              ? enforcement.suspension.description
              : enforcement.description
          }
        />
      </Layout>
    </StyledLayout>
  );
};
