import * as React from "react";

import {
  Background,
  CoreText,
  Display,
  FlexDirection,
  Interactable,
  InteractableType,
  StyledLayout
} from "twitch-core-ui";

import { Enforcement, EnforcementTypes, UserRole } from "aegis/models";
import { EnforcementSuspensionBody, EnforcementWarningBody } from "../enforcement-body";
import { EnforcementHeader } from "../enforcement-header/component";

export interface Props {
  userProfileImage?: string;
  roles?: UserRole[];
}

export interface PublicProps {
  enforcement: Enforcement;
  expanded: boolean;
  expand: () => void;
  collapse: () => void;
}

export class EnforcementCardComponent extends React.Component<Props & PublicProps> {
  public render() {
    const { enforcement, userProfileImage, roles, expanded } = this.props;
    let body;
    let targetUserName;
    let duration;
    if (enforcement.enforcementType === EnforcementTypes.Warning && !!enforcement.warning) {
      body = <EnforcementWarningBody enforcement={enforcement} roles={roles} />;
      targetUserName =
        !!enforcement.warning && !!enforcement.warning.targetUserUsername
          ? enforcement.warning.targetUserUsername
          : "Username not found";
    } else if (enforcement.enforcementType === EnforcementTypes.Suspension && !!enforcement.suspension) {
      body = <EnforcementSuspensionBody enforcement={enforcement} roles={roles} />;
      targetUserName =
        !!enforcement.suspension && !!enforcement.suspension.targetUserUsername
          ? enforcement.suspension.targetUserUsername
          : "Username not found";
      duration = !!enforcement.suspension && !!enforcement.suspension.duration ? enforcement.suspension.duration : 0;
    } else {
      body = <CoreText>Error loading data, more then one type present</CoreText>;
      targetUserName = "";
    }

    if (expanded) {
      return (
        <StyledLayout
          display={Display.Flex}
          margin={{ y: 1, x: 1 }}
          flexDirection={FlexDirection.Column}
          elevation={1}
          background={Background.Alt}
        >
          <Interactable onClick={this.props.collapse} blurAfterClick type={InteractableType.Alpha}>
            <EnforcementHeader
              createdAt={new Date(enforcement.createdAt).toDisplayFormat()}
              targetUserName={targetUserName}
              duration={duration}
              profileImage={userProfileImage}
              content={enforcement.content}
              reason={enforcement.reason}
              open
            />
          </Interactable>
          {body}
        </StyledLayout>
      );
    } else {
      return (
        <StyledLayout
          display={Display.Flex}
          margin={{ y: 1, x: 1 }}
          flexDirection={FlexDirection.Column}
          elevation={1}
          background={Background.Alt}
        >
          <Interactable onClick={this.props.expand} blurAfterClick type={InteractableType.Alpha}>
            <EnforcementHeader
              createdAt={new Date(enforcement.createdAt).toDisplayFormat()}
              targetUserName={targetUserName}
              duration={duration}
              profileImage={userProfileImage}
              content={enforcement.content}
              reason={enforcement.reason}
            />
          </Interactable>
        </StyledLayout>
      );
    }
  }
}
