import * as React from "react";

import gql from "graphql-tag";

import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { Enforcement } from "aegis/models";
import { OffsetPageInfo } from "aegis/models/offset_pageinfo";
import { Color, CoreText, Display, Layout, LoadingSpinner } from "twitch-core-ui";
import { EnforcementComponent } from "./component";

export type Response = {
  enforcements: {
    enforcements: Enforcement[];
    pageInfo: OffsetPageInfo;
  };
};

export const ENFORCEMENTS_PER_PAGE = 25;
export const FETCH_ENFORCEMENTS = gql`
  query enforcements($user: ID, $adminLDAP: String, $limit: Int, $offset: Int) {
    enforcements(user: $user, adminLDAP: $adminLDAP, limit: $limit, offset: $offset) {
      enforcements {
        ...EnforcementFields
      }
      pageInfo {
        offset
        total
      }
    }
  }
  ${Fragments.EnforcementFields}
`;
export interface InputProps {
  user?: string;
  adminLDAP?: string;
  limit?: number;
  offset?: number;
}

export class EnforcementQuery extends Query<Response, InputProps> {}

export class EnforcementView extends React.Component<InputProps> {
  public render() {
    const { user, adminLDAP } = this.props;
    if (!user && !adminLDAP) {
      return (
        <Layout display={Display.Flex} margin={{ y: 2, x: 1 }}>
          <CoreText>Enter a user name or admin LDAP ID</CoreText>
        </Layout>
      );
    }
    return (
      <EnforcementQuery displayName="EnforcementView" query={FETCH_ENFORCEMENTS} variables={this.props}>
        {({ loading, error, data, refetch }) => {
          if (loading) {
            return <LoadingSpinner />;
          }
          if (error) {
            console.error(error);
            return <CoreText color={Color.Error}>Error: {error.message}</CoreText>;
          }
          if (!data) {
            console.error("No Enforcements Found user: " + this.props.user + " admin: " + this.props.adminLDAP);
            return (
              <div>
                No Enforcements Found for {!!this.props.user && "user: " + this.props.user}
                {!!this.props.user && !!this.props.adminLDAP && " and "}
                {!!this.props.adminLDAP && "admin: " + this.props.adminLDAP}
              </div>
            );
          }
          const getEnforcementsForPage = (pageNumber: number) => {
            refetch({ offset: ENFORCEMENTS_PER_PAGE * (pageNumber - 1) });
          };
          return (
            <EnforcementComponent
              enforcements={data.enforcements.enforcements}
              getEnforcementsForPage={getEnforcementsForPage}
              pageInfo={data.enforcements.pageInfo}
            />
          );
        }}
      </EnforcementQuery>
    );
  }
}
