import * as React from "react";

import gql from "graphql-tag";
import { Color, CoreText, Layout, Pagination, PaginationMouseEvent, TextAlign, TextType } from "twitch-core-ui";

import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { Stream } from "aegis/models";
import { LoadingFill } from "../loading-fill";
import { StreamsComponent } from "./component";

export interface PublicProps {
  game?: string;
  grayscale: boolean;
}

export const FETCH_STREAMS_IN_DIRECTORY = gql`
  query streamsInDirectory($game: String!, $limit: Int!, $offset: Int!) {
    streamsInDirectory(game: $game, limit: $limit, offset: $offset) {
      streams {
        viewCount
        duration
        title
        thumbnailURL
        user {
          ...UserFields
        }
      }
      total
    }
  }
  ${Fragments.UserFields}
`;

interface Variables {
  limit: number;
  offset: number;
  game: string;
}

interface Data {
  streamsInDirectory: {
    streams: Stream[];
    total: number;
  };
}

const STREAMS_PER_PAGE = 50;
class GameDirectoryQuery extends Query<Data, Variables> {}

export const GameDirectory: React.SFC<PublicProps> = ({ game, grayscale }) => {
  const [curPage, setCurPage] = React.useState(1);

  if (!game) {
    return (
      <Layout fullWidth textAlign={TextAlign.Center} padding={2}>
        <CoreText type={TextType.H4}>Please input a game name above</CoreText>
      </Layout>
    );
  }
  return (
    <GameDirectoryQuery
      displayName="GameDirectory"
      query={FETCH_STREAMS_IN_DIRECTORY}
      skip={!game}
      variables={{ game: game, limit: STREAMS_PER_PAGE, offset: (curPage - 1) * STREAMS_PER_PAGE }}
    >
      {({ loading, error, data }) => {
        if (error) {
          console.error("Error looking up streams", error);
          return <CoreText color={Color.Error}> Error looking up streams </CoreText>;
        }
        if (!data) {
          return <CoreText color={Color.Error}> No streams found for {game} </CoreText>;
        }

        if (loading) {
          return <LoadingFill />;
        }
        if (data.streamsInDirectory.streams.length === 0) {
          return (
            <Layout fullWidth textAlign={TextAlign.Center} padding={2}>
              <CoreText type={TextType.H4}>
                No streams found in game {game}. Make sure the name is an exact match
              </CoreText>
            </Layout>
          );
        }

        return (
          <Layout>
            <StreamsPagination currentPage={curPage} total={data.streamsInDirectory.total} update={setCurPage} />
            <StreamsComponent streams={data.streamsInDirectory.streams} grayscale={grayscale} />
            <StreamsPagination currentPage={curPage} total={data.streamsInDirectory.total} update={setCurPage} />
          </Layout>
        );
      }}
    </GameDirectoryQuery>
  );
};

const StreamsPagination: React.SFC<{ currentPage: number; total: number; update: (n: number) => void }> = ({
  currentPage,
  total,
  update
}) => {
  return (
    <Pagination
      currentPage={currentPage}
      totalPages={Math.ceil(total / STREAMS_PER_PAGE)}
      onClickIndex={(e: PaginationMouseEvent<HTMLDivElement>): void => {
        update(e.goToPage);
      }}
      onClickNext={() => update(currentPage + 1)}
      onClickPrevious={() => update(currentPage - 1)}
      nextPageButtonAriaLabel="Next Page"
      previousPageButtonAriaLabel="Previous Page"
    />
  );
};
