import * as React from "react";

import gql from "graphql-tag";
import { Mutation } from "react-apollo";

import { apolloErrorHandler, errorNotification } from "aegis/functionality/error-notifications";
import { User, UserRole } from "aegis/models";
import { Button, ButtonType, Tooltip } from "twitch-core-ui";

export enum QuickStrikeReasons {
  TosBanEvasion = "tos_ban_evasion",
  OffensiveUsername = "offensive_username",
  SexualContent = "sexual_content",
  Harm = "harm",
  Miscategorized = "miscategorized"
}

const descriptionPrefix: (
  content: string,
  reason: string,
  ipBan: boolean,
  clearImages: boolean,
  points: number
) => string = (content, reason, ipBan, clearImages, points) => {
  return (
    `Content: ${content}\n` +
    `Reason: ${reason}\n` +
    `IP Block: ${ipBan}\n` +
    "Suspension: " +
    (points >= 100 ? "indefinite" : `${points} Day(s)`) +
    "\n" +
    `Cleared Images: ${clearImages}\n` +
    "----------------------------------------\n\n"
  );
};

const getInputForReason: (reason: QuickStrikeReasons, targetUserID: string) => QuickStrikeInput | null = (
  reason,
  targetUserID
) => {
  switch (reason) {
    case QuickStrikeReasons.TosBanEvasion:
      return {
        input: {
          targetUserID: targetUserID,
          reason: reason,
          detailedReason: reason,
          content: "general",
          tosBan: true,
          ipBan: true,
          clearImages: false,
          points: 100,
          description:
            descriptionPrefix("general", reason, true, false, 100) +
            "[Directory Strike] Site ban evasion\n\nSee Suspending Specialist for POC"
        }
      };
    case QuickStrikeReasons.OffensiveUsername:
      return {
        input: {
          targetUserID: targetUserID,
          reason: reason,
          detailedReason: "inappropriate_username",
          content: "general",
          tosBan: true,
          ipBan: false,
          clearImages: false,
          points: 100,
          description:
            descriptionPrefix("general", reason, false, false, 100) +
            "[Directory Strike] Inappropriate or offensive username\n\nSee Suspending Specialist for POC"
        }
      };
    case QuickStrikeReasons.SexualContent:
      return {
        input: {
          targetUserID: targetUserID,
          reason: reason,
          detailedReason: "sexually_explicit_content",
          content: "live",
          tosBan: true,
          ipBan: false,
          clearImages: false,
          points: 7,
          description:
            descriptionPrefix("live", reason, false, false, 7) +
            "[Directory Strike] Porn or explicit sex content\n\nSee Suspending Specialist for POC"
        }
      };
    case QuickStrikeReasons.Harm:
      return {
        input: {
          targetUserID: targetUserID,
          reason: reason,
          detailedReason: "murder_or_maiming",
          content: "live",
          tosBan: true,
          ipBan: true,
          clearImages: false,
          points: 100,
          description:
            descriptionPrefix("live", reason, true, false, 100) +
            "[Directory Strike] Extreme violence, gore, death or disturbing content\n\nSee Suspending Specialist for POC"
        }
      };
    case QuickStrikeReasons.Miscategorized:
      return {
        input: {
          targetUserID: targetUserID,
          reason: reason,
          detailedReason: "trending_intentional_miscategorization",
          content: "live",
          tosBan: true,
          ipBan: false,
          clearImages: false,
          points: 1,
          description:
            descriptionPrefix("live", reason, false, false, 1) +
            "[Directory Strike] User intentionally streaming unrelated content in this category\n\nSee Suspending Specialist for POC"
        }
      };
    default:
      return null;
  }
};

export const QUICK_STRIKE_MUTATION = gql`
  mutation createSuspension($input: CreateSuspensionInput!) {
    createSuspension(input: $input) {
      targetUser {
        id
        isBanned
      }
      enforcementType
    }
  }
`;

type QuickStrikeInput = {
  input: {
    targetUserID: string;
    content: string;
    reason: string;
    detailedReason: string;
    description: string;
    tosBan: boolean;
    ipBan: boolean;
    clearImages: boolean;
    points: number;
  };
};

type QuickStrikeResponse = {
  createSuspension: {
    targetUser: {
      id: string;
      isBanned: boolean;
    };
  };
};

export interface StrikeButtonProps {
  targetUser: User;
  selectedReason: QuickStrikeReasons;
}

export class QuickStrikeMutation extends Mutation<QuickStrikeResponse, QuickStrikeInput> {}

export const QuickStrikeButton: React.SFC<StrikeButtonProps> = ({ selectedReason, targetUser, children }) => {
  return (
    <QuickStrikeMutation mutation={QUICK_STRIKE_MUTATION} onError={apolloErrorHandler("Failed to strike user")}>
      {(resolveReportQuery, { data, loading, error }) => {
        const onClick = () => {
          const input = getInputForReason(selectedReason, targetUser.id);
          if (input) {
            resolveReportQuery({ variables: input });
          } else {
            errorNotification("No input selected...?");
          }
        };

        const isPartner = targetUser.roles && targetUser.roles.includes(UserRole.Partner);

        let type = ButtonType.Default;
        if (targetUser.isBanned) {
          type = ButtonType.Alert;
        } else if (!!data && !error) {
          type = ButtonType.Success;
        }
        const button = (
          <Button onClick={onClick} disabled={loading || isPartner || targetUser.isBanned} type={type}>
            {children}
          </Button>
        );

        if (isPartner) {
          return <Tooltip label="Partners can not be struck from this tool">{button}</Tooltip>;
        } else {
          return button;
        }
      }}
    </QuickStrikeMutation>
  );
};
